(ns org.euandre.om-auth.controllers.graph.auth
  (:require [om.next :as om]
            [org.euandre.om-auth.db.datomic.user :as db.user]
            [com.wsscode.pathom.core :as pathom]
            [clj-time.core :as time]
            [datomic.api :as d]
            [org.euandre.om-auth.logic.auth :as logic.auth]
            [org.euandre.om-auth.controllers.email :as c-email]
            [org.euandre.om-auth.components :as c]
            [org.euandre.misc.exception :as exception]))

(defmulti virtual-attribute pathom/key-dispatch)
(defmethod virtual-attribute :default [_] ::pathom/continue)

(defmulti entity-reader pathom/entity-dispatch)
(defmethod entity-reader :default [_] ::pathom/continue)

(defmulti mutate om/dispatch)

(defn bad-password! []
  (exception/invalid-input! {::exception/reason ::email-password-mismatch
                             ::exception/reason-str       "Invalid email/password combination."}))

(defn bad-token! []
  (exception/invalid-input! {::exception/reason ::bad-email-token
                             ::exception/reason-str       "Bad email <-> token input."}))

;; Entrypoint controllers

(defn confirm-email! [confirmation-id datomic]
  (let [user (db.user/lookup-confirmation-id confirmation-id (d/db datomic))
        [should-confirm? return-message] (logic.auth/should-confirm-user user)]
    (when should-confirm?
      (db.user/confirm-user! (:user/id user) datomic))
    return-message))

(defn reset-password-page [reset-password-id datomic]
  "<html><body><h1>xablau</h1></body></html>")

(defn reset-password! [reset-password-id {:keys [password confirm-password]} datomic]
  (when-not (= password confirm-password)
    (exception/invalid-input! {::exception/reason     ::password-confirm-password-mismatch
                               ::exception/reason-str "Password and confirm password does not match each other."}))
  (let [{:user/keys [id]} (db.user/lookup-reset-password-id! reset-password-id (d/db datomic))
        hashed-password   (logic.auth/hash-password password)]
    (db.user/update-password-hash! id hashed-password datomic)
    {:ok true
     :message "Password successfully reset."}))

;; Om.Next/Pathom readers

(defmethod virtual-attribute :user/password-hash  [_env] ::pathom/not-found)
(defmethod virtual-attribute :user/refresh-tokens [_env] ::pathom/not-found)

(defmethod virtual-attribute :auth/validate-email
  [{{:keys [db]} ::c/components :as env}]
  (let [email           (pathom/ident-value env)
        validated-email (logic.auth/validate-unique-email email db)]
    (pathom/join validated-email env)))

(defmethod entity-reader :user/by-email
  [{{:keys [db]} ::c/components :as env}]
  (let [email (pathom/ident-value env)
        user (db.user/lookup-by-email email db)]
    (pathom/join user env)))

(defmethod entity-reader :user/by-id
  [{{:keys [db]} ::c/components :as env}]
  (let [user-id (pathom/ident-value env)
        user (db.user/lookup-by-id user-id db)]
    (pathom/join user env)))

;; Effectfull helpers

(defn new-token! []
  (let [randomdata (buddy.core.nonce/random-bytes 160)]
    (buddy.core.codecs/bytes->hex randomdata)))

(defn issue-new-jwt! [user-id scopes datomic config]
  (let [now            (time/now)
        refresh-before (logic.auth/now->refresh-token-expiration now)
        refresh-token  (new-token!)
        token-id       (d/squuid)]
    (db.user/create-refresh-token-for-user! user-id
                                            token-id
                                            (logic.auth/hash-password refresh-token)
                                            refresh-before
                                            datomic)
    (logic.auth/new-wrapped-jwt-payload user-id scopes refresh-token now token-id config)))

(def new-confirmation-id d/squuid)
(def new-reset-token-id d/squuid)

;; Mutations

(defmethod mutate 'auth/register
  [{{:keys [db datomic config]} ::c/components :as request} _key {:user/keys [email password id]}]
  {:action
   #(let [user-id         (d/squuid)
          confirmation-id (new-confirmation-id)
          user            (logic.auth/new-user email password db user-id confirmation-id)]
      (db.user/create-user! user datomic)
      (c-email/send-confirmation-email! email confirmation-id config)
      {:tempids {id user-id}})})

(defmethod mutate 'auth/login
  [{{:keys [db datomic config]} ::c/components} _key {:user/keys [email password]}]
  {:action
   #(let [{:user/keys [confirmation-status password-hash scopes id]} (db.user/lookup-by-email! email db)]
      (when-not (logic.auth/verify-password password password-hash)
        (bad-password!))
      (if (= confirmation-status :confirmation/confirmed)
        (issue-new-jwt! id scopes datomic config)
        (exception/precondition-failed! {::exception/reason ::email-not-confirmed-on-login
                                         ::exception/reason-str "Email should be confirmed to login."})))})

(defmethod mutate 'auth/refresh
  [{{:keys [db datomic config]} ::c/components {:keys [sub]} :identity} _key {:user/keys [refresh-token]}]
  {:action
   #(let [{:user/keys [id scopes refresh-tokens]} (db.user/lookup-by-id! sub db)
          maybe-hashed-refresh-token (some (fn [{:refresh-token/keys [token-hash]}]
                                             (if (logic.auth/verify-password refresh-token token-hash)
                                               token-hash
                                               false))
                                           refresh-tokens)]
      (when-not maybe-hashed-refresh-token
        (bad-token!))
      (db.user/revoke-refresh-token-by-hash! maybe-hashed-refresh-token datomic)
      (issue-new-jwt! id scopes datomic config))})

(defmethod mutate 'auth/change-password
  [{{:keys [db datomic]} ::c/components {:keys [sub]} :identity}
   _key
   {:user/keys [old-password new-password]}]
  {:action
   #(let [{:user/keys [id password-hash]} (db.user/lookup-by-id! sub db)]
      (when-not (logic.auth/verify-password old-password password-hash)
        (bad-password!))
      (db.user/change-password! id password-hash (logic.auth/hash-password new-password) datomic)
      {:tempids {}})})

(defmethod mutate 'auth/add-email
  [{{:keys [db datomic]} ::c/components} _key {:user/keys [id email]}]
  {:action
   #(let [{:user/keys [id]} (db.user/lookup-by-id! id db)
          validated-email   (logic.auth/assert-good-email email db)]
      (db.user/assoc-email! id validated-email datomic)
      {:tempids {}})})

(defmethod mutate 'auth/remove-email
  [{{:keys [db datomic]} ::c/components} _key {:user/keys [email]}]
  {:action
   #(let [{:user/keys [id emails]} (db.user/lookup-by-email! email db)]
      (when (= 1 (count emails))
        (exception/conflict! {::exception/reason     ::should-have-at-least-one-id
                              ::exception/reason-str "Can't remove email because user must have at least 1 registered email."}))
      (db.user/dissoc-email! id email datomic)
      {:tempids {}})})

(defmethod mutate 'auth/logout
  [{{:keys [datomic]} ::c/components {:keys [jti]} :identity} _key _params]
  {:action
   #(do
      (db.user/revoke-refresh-tokens-by-id! #{jti} datomic)
      {:tempids {}})})

(defmethod mutate 'auth/logout-others
  [{{:keys [db datomic]} ::c/components {:keys [jti sub]} :identity} _key _params]
  {:action
   #(let [ids (filter (partial not= jti) (db.user/tokens-id-by-user-id sub db))]
      (db.user/revoke-refresh-tokens-by-id! ids datomic)
      {:tempids {}})})

(defmethod mutate 'auth/logout-all
  [{{:keys [db datomic]} ::c/components {:keys [sub]} :identity} _key _params]
  {:action
   #(let [ids (db.user/tokens-id-by-user-id sub db)]
      (db.user/revoke-refresh-tokens-by-id! ids datomic)
      {:tempids {}})})

(defmethod mutate 'auth/forgot-password
  [{{:keys [db datomic config]} ::c/components} _key {:user/keys [email]}]
  {:action
   (fn []
     (when-let [{:user/keys [id]} (db.user/lookup-by-email email db)]
       (let [reset-password-id (new-reset-token-id)]
         (db.user/register-reset-password-token! id reset-password-id datomic)
         (c-email/send-reset-password-email! email reset-password-id config)))
     {:tempids {}})})
