(ns org.euandre.misc.core
  "Miscelaneous utility (pure) functions/language extensions. A.K.A. \"things that should be in `clojure.core`\".")

(defn map-keys
  "Applies function `f` to every key in map `m`.

  => (map-keys {:a 1 :b 2})
  {\":a\" 1 \":b\" 2}"
  [f m]
  (into {}
        (map (fn [[k v]]
               [(f k) v])
             m)))

(defn map-vals
  "Applies function `f` to every value in map `m`.

  => (map-vals {:a 1 :b 2})
  {:a \"1\" :b \"2\"}"
  [f m]
  (into {}
        (map (fn [[k v]]
               [k (f v)])
             m)))

(defn- recurse-on-pairs-of-arguments
  "Calls function `f` with map `m` as input. Takes pairs of arguments from sequence `more`. Expects `more` to have an even number of elements.

  Function `f` receives the map `m` as it's first argument, the first and second elements of the tuple as it's second and third argument, and returns a map.
  (fn [map t1 t2] ...)

  ;; hypothetical function that assocs only namespaces keys to the map:
  => (recurse-on-pairs-of-arguments (fn [map key value]
                                      (if (namespace key)
                                        (assoc map key value)
                                        map))
                                    {:ns/a 1}
                                    :ns/b 2
                                    :c 3
                                    :ns/d 4)
  #:ns{:a 1 :b 2 :d 4}"
  [f m & more]
  (assert (even? (count more))
          (str "#'" (ns-name *ns*) "/recurse-on-pairs-of-arguments expects even number of arguments after map/vector, found odd number."))
  (loop [m m
         more more]
    (if (empty? more)
      m
      (let [[k v & even-more] more]
        (recur (f m k v)
               even-more)))))

(defn assoc-if
  "Associates only truthy values.

  => (assoc-if {:a 1} :b nil :c false :d :truthy)
  {:a 1 :d :truthy}"
  [m k v & more]
  (apply recurse-on-pairs-of-arguments
         (fn [m k v]
           (cond-> m v (assoc k v)))
         m
         k
         v
         more))

(defn assoc-in-if
  "Associates-in only truthy values. Same as `assoc-if`, but for nested maps.

  => (misc/assoc-in-if {} [:a] 1 [:b] nil [:c] (fn-that-returns-false) [:d] :truthy [:e :f] true)
  {:a 1 :d :truthy :e {:f true}}"
  [m ks v & more]
  (apply recurse-on-pairs-of-arguments
         (fn [m ks v]
           (cond-> m v (assoc-in ks v)))
         m
         ks
         v
         more))

(defn dissoc-in
  "Dissociates a value in a nested data structure.

  => (misc/dissoc-in {:a {:b 1 :c 2}} [:a :c])
  {:a {:b 1}}"
  [m ks & kss]
  (if (empty? ks)
    m
    (update-in m (butlast ks) #(dissoc % (last ks)))))

(defn dissoc-if
  "Dissociates only if pred is truthy.

  => (with-redefs [should-dissoc-key? (constantly true)]
       (misc/dissoc-if {:a 1 :b 2} :b (should-dissoc-key? :b)))
  {:a 1}"
  [m k pred & more]
  (apply recurse-on-pairs-of-arguments
         (fn [m k pred]
           (cond-> m pred (dissoc k)))
         m
         k
         pred
         more))

(defn dissoc-in-if
  "Dissociates-in only if pred is truthy. Same as `dissoc-if` but for nested maps.

  => (dissoc-in-if {:a {:b 1 :c 2}} [:a :b] true [:a :c] false)
  {:a {:c 2}}"
  [m ks pred & more]
  (apply recurse-on-pairs-of-arguments
         (fn [m ks pred]
           (cond-> m pred (dissoc-in ks)))
         m
         ks
         pred
         more))

(defn update-if
  "Updates only truthy values. If you want to update also non-truthy values, just use `#'clojure.core/update`.

  => (update-if {:a 1 :b nil} :a inc :b inc)
  {:a 2 :b nil}"
  [m k f & more]
  (apply recurse-on-pairs-of-arguments
         (fn [m k f]
           (cond-> m (get m k) (update k f)))
         m
         k
         f
         more))

(defn update-in-if
  "Updates-in only truthy values. Same as `update-if` but for nested maps.

  If you want to update-in also non-truthy values, just use `#'clojure.core/update-in`.

  => (update-in-if {:a {:b 2 :c false}} [:a :b] inc [:a :c] dec)
  {:a {:b 3 :c false}}"
  [m ks f & more]
  (apply recurse-on-pairs-of-arguments
         (fn [m ks f]
           (cond-> m (get-in m ks) (update-in ks f)))
         m
         ks
         f
         more))
