(ns org.euandre.misc.core
  "Miscelaneous utility (pure) functions. A.K.A. \"things that should be in `clojure.core`\".")

(defn map-keys
  "Applies f to every key in m."
  [f m]
  (into {}
        (map (fn [[k v]]
               [(f k) v])
             m)))

(defn map-vals
  "Applies f to every value in m."
  [f m]
  (into {}
        (map (fn [[k v]]
               [k (f v)])
             m)))

(defn- recurse-on-pairs-of-arguments [f m & more]
  (assert (even? (count more))
          "#'pires.logic.misc/recurse-on-pairs-of-arguments expects even number of arguments after map/vector, found odd number.")
  (loop [m m
         more more]
    (if (empty? more)
      m
      (let [[k v & even-more] more]
        (recur (f m k v)
               even-more)))))

(defn assoc-if
  "Associates only truthy values."
  [m k v & more]
  (apply recurse-on-pairs-of-arguments
         (fn [m k v]
           (cond-> m v (assoc k v)))
         m
         k
         v
         more))

(defn assoc-in-if
  "Associates-in only truthy values."
  [m ks v & more]
  (apply recurse-on-pairs-of-arguments
         (fn [m ks v]
           (cond-> m v (assoc-in ks v)))
         m
         ks
         v
         more))

(defn dissoc-in
  "Dissociates a value in a nested data structure."
  [m ks & kss]
  (if (empty? ks)
    m
    (update-in m (butlast ks) #(dissoc % (last ks)))))

(defn dissoc-if
  "Dissociates only if pred is truthy."
  [m k pred & more]
  (apply recurse-on-pairs-of-arguments
         (fn [m k pred]
           (cond-> m pred (dissoc k)))
         m
         k
         pred
         more))

(defn dissoc-in-if
  "Dissociates-in only if pred is truthy."
  [m ks pred & more]
  (apply recurse-on-pairs-of-arguments
         (fn [m ks pred]
           (cond-> m pred (dissoc-in ks)))
         m
         ks
         pred
         more))

(defn update-if
  "Updates only truthy values. If you want to update also non-truthy values, just use `#'clojure.core/update`."
  [m k f & more]
  (apply recurse-on-pairs-of-arguments
         (fn [m k f]
           (cond-> m (get m k) (update k f)))
         m
         k
         f
         more))

(defn update-in-if
  "Updates-in only truthy values. If you want to update-in also non-truthy values, just use `#'clojure.core/update-in`."
  [m ks f & more]
  (apply recurse-on-pairs-of-arguments
         (fn [m ks f]
           (cond-> m (get-in m ks) (update-in ks f)))
         m
         ks
         f
         more))
