(ns org.euandre.http.interceptors
  (:require [buddy.auth.middleware :as auth.middleware]
            [clojure.spec.alpha :as s]
            [io.pedestal.http.body-params :as body-params]
            [io.pedestal.interceptor :as i]
            [io.pedestal.interceptor.chain :as i.chain]
            [org.euandre.http.exception :as exception]
            [io.pedestal.log :as log]
            [org.euandre.misc.core :as misc]))

(defn- terminate-with [context response]
  (i.chain/terminate (assoc context :response response)))

(def valid-query?
  (i/interceptor
   {:name  ::valid-query?
    :enter (fn valid-query?-fn [{{:keys [transit-params edn-params]} :request :as context}]
             (let [params (or transit-params edn-params)]
               (if-let [error-reason (cond
                                       (not (map? params))         "Body isn't an edn map."
                                       (not (vector? (:q params))) "[:q] in body isn't an edn vector.")]
                 (exception/invalid-input! {:exception/reason     ::bad-query
                                            :exception/reason-str error-reason
                                            :exception/body       params})
                 (assoc-in context [:request :query] params))))}))

(def body-params
  (body-params/body-params
   (body-params/default-parser-map
     :edn-options {:readers (merge default-data-readers *data-readers*)})))

(def catch
  (i/interceptor
   {:name  ::catch
    :error (fn [context exception]
             (let [{:exception/keys [status-code reason-str reason]} (ex-data exception)]
               (log/error :exception exception)
               (terminate-with context
                               {:status status-code
                                :body   (misc/assoc-if {:error (or (.getMessage exception)
                                                                   "Server Error")}
                                                       :reason reason
                                                       :reason-str reason-str)})))}))

(def authentication-interceptor
  "Port of buddy-auth's wrap-authentication middleware."
  (i/interceptor
   {:name  ::authenticate
    :enter (fn [context]
             (let [jws-backend (get-in context [:request :components :crypto])]
               (update context :request auth.middleware/authentication-request jws-backend)))}))
