(ns ^:figwheel-no-load erinite.core
  (:require-macros
    [reagent.ratom :refer [reaction]]) 
  (:require
    [figwheel.client :as figwheel :include-macros true]
    [cljs.repl :as repl :include-macros true]
    [weasel.repl :as weasel]
    [devtools.core :as devtools]
    [reagent.core :as r]
    [re-frame.core :refer [register-handler
                           register-sub
                           path
                           dispatch 
                           dispatch-sync
                           subscribe]]
    [erinite.signals :as signals]))

(enable-console-print!)

(figwheel/watch-and-reload
  :websocket-url "ws://localhost:3449/figwheel-ws"
  :jsload-callback (fn [] (r/force-update-all)))
(weasel/connect "ws://localhost:9001" :verbose true)
(devtools/install!)


(defn css
  [class-map]
  (clojure.string/join
    " "
    (for [[class-name included?] class-map]
      (when included?
        (name class-name)))))

(register-handler
  :_tools/init
  (path [:_tools])
  (fn [db [_ data]]
    (merge db data)))

(register-handler
  :_tools/toggle-open
  (path [:_tools :open?])
  (fn [value _]
    (not value)))

(register-handler
  :_tools/toggle-collapse
  (path [:_tools :state :collapse])
  (fn [collapses [_ path]]
    (assoc collapses path (not (get collapses path)))))

(register-handler
  :_tools/select-tool
  (path [:_tools :active-tool])
  (fn [value [_ tool]]
    tool))

(register-handler
  :_tools/log-signal
  (path [:_tools :signals])
  (fn [signals [_ signal events]]
    (conj signals [signal (into [] events)])))


(register-sub
  :_tools/open?
  (fn [db _]
    (reaction (get-in @db [:_tools :open?]))))

(register-sub
  :_tools/active-tool
  (fn [db _]
    (reaction (get-in @db [:_tools :active-tool]))))

(register-sub
  :_tools/state
  (fn [db _]
    (reaction @db)))

(register-sub
  :_tools/collapsed?
  (fn [db [_ path]]
    (if (nil? path)
      (reaction nil)
      (reaction (get-in @db [:_tools :state :collapse path])))))

(register-sub
  :_tools/signal-list
  (fn [db _]
    (reaction (get-in @db [:_tools :signals]))))


(defn render-data
  [data path]
  (let [collapsed?  (subscribe [:_tools/collapsed? path])
        collapse!   (fn [e]
                      (dispatch [:_tools/toggle-collapse path])
                      (.stopPropagation e))]
    (fn [data path]
      (cond
        (map? data)
          [:div.data-type-map 
            {:class (css {:collapsed @collapsed?})
             :on-click collapse!}
            "{"
            (for [[key value] data]
              [:div.data-map-entry
                {:key (str key)}
                [:div.data-map-key [render-data key nil]]
                [:div.data-map-value [render-data value (conj path key)]]])
            "}"]
        (vector? data)
          [:div.data-type-vector
            "["
            (map-indexed
              (fn [index value]
                [:div.data-vec-value
                  {:key index}
                  [render-data value (conj path index)]])
              data)
            "]"]
        (string? data)
          [:div.data-type-string
            [:div.data-string-value
              {:on-click collapse!}
              (if @collapsed? "\"…\"" data)]]
        (number? data)
          [:div.data-type-number
            [:div.data-nubmer-value data]]
        (keyword? data)
          [:div.data-type-keyword
            [:div.data-keyword-value (str data)]]
        :else
          [:div.data-type-unknown
            [:div.data-unknown-value (str data)]]))))

(defn render-signals
  []
  (let [signals (subscribe [:_tools/signal-list])]
    (fn []
      [:div
        (map-indexed
          (fn [index [signal events]]
            [:div.signal
              {:key (str signal index)}
              [:div.signal-name
                (str signal)]
              [:div.events-list
                (map-indexed
                  (fn [index event]
                    [:div.event
                     {:key index}
                     (str event)])
                  events)]])
          @signals)])))

(defn tooling
  [app]
  (let [open? (subscribe [:_tools/open?])
        state (subscribe [:_tools/state])
        tool  (subscribe [:_tools/active-tool])]
    (fn [app]
      [:div.erinite-tooling 
       {:class (css {:erinite-open @open? })}
       [:div.erinite-tools
        [:div.erinite-toggle
         {:href "#"
          :on-click #(dispatch [:_tools/toggle-open])}
         (when @open? "Erinite Tooling")]
        (when @open?
          [:div
            [:div.erinite-tools-menu
              [:div.erinite-tools-menu-item
                {:class (when (= @tool :state) :selected)
                 :on-click #(dispatch [:_tools/select-tool :state])}
                "State"]
              [:div.erinite-tools-menu-item
                {:class (when (= @tool :signals) :selected)
                 :on-click #(dispatch [:_tools/select-tool :signals])}
                "Signals"]]
            (case @tool
              :state
                [:div.erinite-tool-state
                  [render-data (dissoc @state :_tools) []]]
              :signals
                [:div.erinite-tool-signals
                  [render-signals]])])]
       [:div.erinite-app
        [app]]])))



(set! signals/log-signal!
      #(dispatch [:_tools/log-signal %1 %2]))
#_(core/init! tooling)
(dispatch-sync [:_tools/init {:open? false
                              :signals []
                              :active-tool :state}])

