;; Compiled from src/funcgo/main.go
(ns
 funcgo.main
 (:gen-class)
 (:require
  [clojure.java.io :as io]
  [clojure.pprint :as pprint]
  [clojure.string :as string]
  [clojure.tools.cli :as cli]
  [funcgo.core :as core])
 (:import
  (java.io BufferedWriter File StringWriter IOException)
  (jline.console ConsoleReader)))

(set! *warn-on-reflection* true)

(let
 [command-line-options
  [["-r" "--repl" "start a Funcgo interactive console"]
   ["-s" "--sync" "No asynchronous channel constructs"]
   ["-n" "--nodes" "print out the parse tree that the parser produces"]
   ["-u" "--ugly" "do not pretty-print the Clojure"]
   ["-f" "--force" "Force compiling even if not out-of-date"]
   ["-a"
    "--ambiguity"
    "print out all matched parse trees to diagnose ambiguity"]
   ["-h" "--help" "print help"]]]
 (defn-
  pretty-print
  [obj writer]
  (let
   [orig-dispatch pprint/*print-pprint-dispatch*]
   (pprint/with-pprint-dispatch
    (fn
     [o]
     (do
      (let
       [met (meta o)]
       (when
        met
        (do
         (print "^")
         (if
          (= (count met) 1)
          (if
           (met :tag)
           (orig-dispatch (met :tag))
           (if
            (= (met :private) true)
            (orig-dispatch :private)
            (orig-dispatch met)))
          (orig-dispatch met))
         (print " ")
         (pprint/pprint-newline :fill))))
      (orig-dispatch o)))
    (pprint/pprint obj writer))))
 (defn-
  write-pretty-to
  [clj-text ^BufferedWriter writer]
  (do
   (doseq
    [expr (read-string (str "[" clj-text "]"))]
    (do (pretty-print expr writer) (. writer (newLine))))
   (. writer (close))))
 (defn-
  compile-expression
  [in-path fgo-text]
  (let
   [clj-text
    (core/Parse in-path fgo-text :expr)
    str-writer
    (StringWriter.)
    writer
    (BufferedWriter. str-writer)]
   (write-pretty-to clj-text writer)
   (. str-writer (toString))))
 (defn-
  new-console-reader
  []
  (let
   [console-reader (ConsoleReader.)]
   (. console-reader (setPrompt "fgo=>     "))
   console-reader))
 (defn-
  repl
  []
  (let
   [^ConsoleReader console-reader (new-console-reader)]
   (loop
    []
    (let
     [fgo-text (. console-reader (readLine))]
     (when
      (not (string/blank? fgo-text))
      (do
       (try
        (let
         [clj-text (first (core/Parse "repl.go" fgo-text :expr))]
         (println "Clojure: " clj-text)
         (println "Result:  " (eval (read-string clj-text))))
        (catch Exception e (println e)))
       (println)))
     (when (not= fgo-text nil) (recur))))))
 (defn
  Compile-string
  [in-path fgo-text]
  (let
   [clj-text
    (core/Parse in-path fgo-text)
    str-writer
    (StringWriter.)
    writer
    (BufferedWriter. str-writer)]
   (write-pretty-to clj-text writer)
   (. str-writer (toString))))
 (defn-
  compile-file
  ([^File in-file ^File root opts]
   (let
    [split-root
     (re-matches #"([^\.]+)(\.[a-z]+)?(\.gos?)" (. in-file getPath))]
    (when
     (not (nil? split-root))
     (let
      [[_ in-path suffix-extra suffix] split-root]
      (compile-file
       in-file
       root
       (str in-path suffix)
       opts
       (if (nil? suffix-extra) "" suffix-extra))))))
  ([^File in-file ^File root in-path opts suffix-extra]
   (let
    [out-file
     (io/file
      (string/replace
       in-path
       #"\.go(s?)$"
       (str ".clj$1" suffix-extra)))]
    (when
     (or
      (opts :force)
      (< (. out-file (lastModified)) (. in-file (lastModified))))
     (let
      [prefix-len
       (. (. root (getAbsolutePath)) (length))
       relative
       (subs (. in-file (getAbsolutePath)) (+ prefix-len 1))]
      (println "  " relative "...")
      (let
       [fgo-text
        (slurp in-file)
        lines
        (count (filter (fn* [p1__30#] (= p1__30# \newline)) fgo-text))
        start
        (if (= suffix-extra "") :sourcefile :nonpkgfile)]
       (try
        (let
         [begin-time
          (System/currentTimeMillis)
          ^String clj-text
          (core/Parse
           relative
           fgo-text
           start
           (opts :nodes)
           (opts :sync)
           (opts :ambiguity))
          duration
          (- (System/currentTimeMillis) begin-time)
          writer
          (io/writer out-file)]
         (. writer (write (str ";; Compiled from " in-file "\n")))
         (if
          (opts :ugly)
          (do (. writer (write clj-text)) (. writer (close)))
          (write-pretty-to clj-text writer))
         (if
          (= (. out-file (length)) 0)
          (do
           (. out-file (delete))
           (println "\t\tERROR: No output created."))
          (do
           (println
            "\t\t-->"
            (. out-file (getPath))
            (long (/ (* 1000.0 lines) duration))
            "lines/s")
           (when
            (< (/ (. out-file length) (. in-file length)) 0.4)
            (println
             "WARNING: Output file is only"
             (long (/ (* 100 (. out-file length)) (. in-file length)))
             "% the size of the input file")))))
        (catch
         IOException
         e
         (println
          "Parsing "
          relative
          " failed:\n"
          (. e (getMessage)))))))))))
 (defn-
  compile-tree
  [^File root opts]
  (do
   (println (. root (getName)))
   (doseq
    [f (file-seq root)]
    (let
     [^File in-file f]
     (try
      (compile-file in-file root opts)
      (catch IOException e (println (. e (getMessage))))
      (catch Exception e (. e (printStackTrace))))))))
 (defn-
  print-error
  [cmd-line]
  (do
   (println)
   (when (cmd-line :errors) (println (cmd-line :errors)))
   (println "USAGE:  fgoc [options] path ...")
   (println "options:")
   (println (cmd-line :summary))))
 (defn
  Compile
  [& args]
  (let
   [cmd-line
    (cli/parse-opts args command-line-options)
    other-args
    (cmd-line :arguments)
    opts
    (cmd-line :options)
    here
    (io/file ".")]
   (if
    (or (cmd-line :errors) (opts :help))
    (println (cmd-line :summary))
    (do
     (if
      (not (seq other-args))
      (do
       (println "Missing directory or file argument.")
       (print-error cmd-line))
      (doseq
       [arg other-args]
       (let
        [file (io/file arg)]
        (if
         (. file isDirectory)
         (compile-tree file opts)
         (try
          (compile-file file here opts)
          (catch Exception e (println "\n" (. e (getMessage)))))))))
     (when (opts :repl) (repl))))))
 (defn -main [& args] (apply Compile args)))

