;; Compiled from src/funcgo/main.go
(ns
 funcgo.main
 (:gen-class)
 (:require
  [clojure.java.io :as io]
  [clojure.pprint :as pprint]
  [clojure.string :as string]
  [clojure.tools.cli :as cli]
  [funcgo.core :as core])
 (:import
  (java.io BufferedWriter File StringWriter)
  (jline.console ConsoleReader)))

(set! *warn-on-reflection* true)

(def
 ^:private command-line-options
 [["-r" "--repl" "start a Funcgo interactive console"]
  ["-n" "--nodes" "print out the parse tree that the parser produces"]
  ["-u" "--ugly" "do not pretty-print the Clojure"]
  ["-f" "--force" "Force compiling even if not out-of-date"]
  ["-h" "--help" "print help"]])

(defn-
 pretty-print
 [obj writer]
 (let
  [orig-dispatch pprint/*print-pprint-dispatch*]
  (pprint/with-pprint-dispatch
   (fn
    [o]
    (do
     (let
      [met (meta o)]
      (when
       met
       (do
        (print "^")
        (if
         (= (count met) 1)
         (if
          (met :tag)
          (orig-dispatch (met :tag))
          (if
           (= (met :private) true)
           (orig-dispatch :private)
           (orig-dispatch met)))
         (orig-dispatch met))
        (print " ")
        (pprint/pprint-newline :fill))))
     (orig-dispatch o)))
   (pprint/pprint obj writer))))

(defn-
 write-pretty-to
 [clj-text ^BufferedWriter writer]
 (do
  (doseq
   [expr (read-string (str "[" clj-text "]"))]
   (do (pretty-print expr writer) (. writer (newLine))))
  (. writer (close))))

(defn-
 compile-expression
 [in-path fgo-text]
 (let
  [clj-text
   (core/Parse in-path fgo-text :expr)
   str-writer
   (StringWriter.)
   writer
   (BufferedWriter. str-writer)]
  (write-pretty-to clj-text writer)
  (. str-writer (toString))))

(defn-
 new-console-reader
 []
 (let
  [console-reader (ConsoleReader.)]
  (. console-reader (setPrompt "fgo=>     "))
  console-reader))

(defn-
 repl
 []
 (let
  [^ConsoleReader console-reader (new-console-reader)]
  (loop
   []
   (let
    [fgo-text (. console-reader (readLine))]
    (when
     (not (string/blank? fgo-text))
     (do
      (try
       (let
        [clj-text (first (core/Parse "repl.go" fgo-text :expr))]
        (println "Clojure: " clj-text)
        (println "Result:  " (eval (read-string clj-text))))
       (catch Exception e (println e)))
      (println)))
    (when (not= fgo-text nil) (recur))))))

(defn
 Compile-string
 [in-path fgo-text]
 (let
  [clj-text
   (core/Parse in-path fgo-text)
   str-writer
   (StringWriter.)
   writer
   (BufferedWriter. str-writer)]
  (write-pretty-to clj-text writer)
  (. str-writer (toString))))

(defn-
 compile-file
 [^File in-file ^File root opts]
 (let
  [in-path
   (. in-file (getPath))
   out-file
   (io/file (string/replace in-path #"\.go(s?)$" ".clj$1"))]
  (when
   (or
    (opts :force)
    (< (. out-file (lastModified)) (. in-file (lastModified))))
   (let
    [prefix-len
     (. (. root (getAbsolutePath)) (length))
     relative
     (subs (. in-file (getAbsolutePath)) (+ prefix-len 1))]
    (print "  " relative " ")
    (let
     [^String clj-text
      (core/Parse relative (slurp in-file) :sourcefile (opts :nodes))
      writer
      (io/writer out-file)]
     (. writer (write (str ";; Compiled from " in-file "\n")))
     (if
      (opts :ugly)
      (do (. writer (write clj-text)) (. writer (close)))
      (write-pretty-to clj-text writer))
     (if
      (= (. out-file (length)) 0)
      (do
       (. out-file (delete))
       (println "\t\tERROR: No output created."))
      (do
       (println "\t\t-->" (. out-file (getPath)))
       (when
        (< (/ (. out-file length) (. in-file length)) 0.4)
        (println
         "WARNING: Output file is only"
         (long (/ (* 100 (. out-file length)) (. in-file length)))
         "% the size of the input file")))))))))

(defn-
 compile-tree
 [^File root opts]
 (do
  (println (. root (getName)))
  (doseq
   [f (file-seq root)]
   (let
    [^File ff f name (. ff getName)]
    (try
     (when
      (or (. name (endsWith ".go")) (. name (endsWith ".gos")))
      (compile-file ff root opts))
     (catch
      Exception
      e
      (println "\n    Compile failed: " (. e (getMessage)))))))))

(defn-
 print-error
 [cmd-line]
 (do
  (println)
  (when (cmd-line :errors) (println (cmd-line :errors)))
  (println "USAGE:  fgoc [options] path ...")
  (println "options:")
  (println (cmd-line :summary))))

(defn
 Compile
 [& args]
 (let
  [cmd-line
   (cli/parse-opts args command-line-options)
   other-args
   (cmd-line :arguments)
   opts
   (cmd-line :options)
   here
   (io/file ".")]
  (if
   (or (cmd-line :errors) (opts :help))
   (println (cmd-line :summary))
   (do
    (if
     (not (seq other-args))
     (do
      (println "Missing directory or file argument.")
      (print-error cmd-line))
     (doseq
      [arg other-args]
      (let
       [file (io/file arg)]
       (if
        (. file isDirectory)
        (compile-tree file opts)
        (try
         (compile-file file here opts)
         (catch Exception e (println "\n" (. e (getMessage)))))))))
    (when (opts :repl) (repl))))))

(defn- -main [& args] (apply Compile args))

