(ns duct.test
  (:require [clojure.java.io :as io]
            [integrant.core :as ig]))

(defn- load-config []
  (let [f (io/file "duct.edn")]
    (if (.exists f)
      (ig/read-string (slurp f))
      {})))

(defn- default-vars [{:keys [vars] :or {vars {}}}]
  (reduce-kv (fn [m k v]
               (if-some [default (:default v)]
                 (assoc m k default)
                 m))
             {} vars))

(defn- prep [{:keys [config vars profiles]
              :or   {profiles [:test]}}]
  (let [config (or config (load-config))
        vars   (merge (default-vars config) vars)]
    (ig/load-hierarchy)
    (-> (:system config)
        (doto ig/load-namespaces)
        (ig/expand (ig/deprofile profiles))
        (ig/deprofile profiles)
        (ig/bind vars))))

(defn run
  "Run the Duct application in a test environment. Supports the following
  options:

  - `:config`   - the config map to use (default loads from `duct.edn`)
  - `:keys`     - only run this collection of keys (default is all keys)
  - `:profiles` - use these profiles (default is `[:test]`)
  - `:vars`     - bind vars to this map of symbols to values"
  ([] (run {}))
  ([options]
   (let [prepped-config (prep options)]
     (ig/load-namespaces prepped-config)
     (if-some [keys (:keys options)]
       (ig/init prepped-config (set keys))
       (ig/init prepped-config)))))

(defmacro with-system
  "Halt a system after the end of the body. Has the same semantics as
  `with-open`."
  [[sym system] & body]
  `(let [~sym ~system]
     (try ~@body
          (finally (ig/halt! ~sym)))))
