(ns org.dthume.goad.core
  (:require [clojure.core.async :as async]
            [clojure.core.async.impl.protocols :as async-p]))

(def channel?
  "Returns true if argument is a core.async channel"
  (partial satisfies? async-p/Channel))

(defmacro return-throwables
  [& body]
  `(try
     ~@body
     (catch Throwable e# e#)))

(defmacro go?
  "A `go` block which will catch any thrown `Throwable` instances
and yield them as the value of the block channel."
  [& body]
  `(async/go
     (return-throwables ~@body)))

(defmacro thread?
  "A `thread` block which will catch any thrown `Throwable` instances
and yield them as the value of the block channel."
  [& body]
  `(async/thread
     (return-throwables ~@body)))

(defmacro go-loop?
  [bindings & body]
  `(go? (loop ~bindings ~@body)))

(defmacro thread-loop?
  [bindings & body]
  `(thread? (loop ~bindings ~@body)))

(defn rethrow-exceptions
  [maybe-e]
  (if (instance? Throwable maybe-e)
    (throw maybe-e)
    maybe-e))

(defmacro <?
  "Read a value with `<!` from the channel found by evaluating `expr` and, if
it is a `Throwable` instance, then throw it, otherwise return it. Must be used
inside a `go` block."
  [expr]
  `(rethrow-exceptions (async/<! ~expr)))

(defmacro <??
  "Read a value with `<!!` from the channel found by evaluating `expr` and,
if it is an `Throwable` instance, then throw it, otherwise return it. Should
not be used inside a `go` block."
  [expr]
  `(rethrow-exceptions (async/<!! ~expr)))

(defn async-response
  "If `r` has a `:body` which is a core.async channel, then returns that
channel, else returns `nil`."
  [r]
  (let [b (and r (:body r))]
    (when (channel? b)
      b)))

(defn async-middleware
  "Utility for constructing ring middleware out of a pair of `:request` and
`:response` handling functions (such as those found in ring core). Returns
a function which can be used to wrap handlers in the resulting middleware.
Note that neither `:request` and `:response` can itself return a core.async
channel; if you need to write async aware middleware then you don't need this
function!"
  [& {:keys [request response]
      :or {request identity response nil}}]
  (fn [handler]
    (fn [req]
      (let [resp (-> req request handler)]
        (if (and response (async-response resp))
          (update-in resp [:body] (partial async/map< response))
          resp)))))

(defmacro <val!
  "Evaluate `expr` and, if it is a channel, read a value from it with `<!`
and yield it as the result, otherwise return the value of `expr`.
Must be used inside a `go` block."
  [expr]
  `(let [v# ~expr]
     (if (channel? v#)
       (async/<! v#)
       v#)))

(defmacro <val?
  "Evaluate `expr` and, if it is a channel, read a value from it with `<?`
and yield it as the result, otherwise return the value of `expr`.
Must be used inside a `go` block."
  [expr]
  `(let [v# ~expr]
     (if (channel? v#)
       (<? v#)
       v#)))

(defmacro <val??
  "Evaluate `expr` and, if it is a channel, read a value from it with `<??`
and yield it as the result, otherwise return the value of `expr`.
Must not be used inside a `go` block."
  [expr]
  `(let [v# ~expr]
     (if (channel? v#)
       (<?? v#)
       v#)))

(defmacro <let?
  "Optimistic mixed blocking and non-blocking version of `let`. Used to write
code which receives functions which may be either blocking or non blocking,
and needs to itself become either blocking or non blocking. This allows
functions to optionally work with core.async without forcing it on clients,
and without requiring two sets of parallel APIs etc. (at the price of a
rather ugly macro).

Establishes `bindings` as if by `let`. If any of the resulting bindings
are a channel, then a `go` block is initiated, the results of any channel
bindings are read inside the block and rebound, and `body` is executed
in the scope of the resulting bindings.

If none of the resulting bindings from the first step are channels
then `body` is executed in the scope of these bindings (no go block)."
  [bindings & body]
  (let [bind-map (apply hash-map bindings)
        bind-keys (-> bind-map keys vec)
        bind-gs (vec (for [k bind-keys] (gensym)))]
    `(let [do-body# (fn ~bind-keys ~@body)
           check-exceptions#
           (fn ~bind-keys
             (if-let [e# (->> ~bind-keys
                              (filter (partial instance? Throwable))
                              first)]
               e#
               (do-body# ~@bind-keys)))
            ~@(mapcat (fn [gs k] [gs (get bind-map k)])
                      bind-gs bind-keys)]
       (if (or ~@(for [gs bind-gs] `(channel? ~gs)))
         (go?
          (<val!
           (check-exceptions# ~@(for [gs bind-gs] `(async/<! ~gs)))))
         (do-body# ~@bind-gs)))))

