(ns mathdoc.endpoint.postcanvas
  (:require [clojure.data.codec.base64 :as b64]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [compojure.core :refer :all]
            ring.util.response
            [taoensso.timbre :as log]
            [integrant.core :as ig]
            [duct.core :as duct])
  (:import [java.io BufferedReader StringReader]
           java.nio.charset.StandardCharsets
           java.util.Date))

;;; implementation

(defn chop-header [s]
  (-> (re-seq #"(data:image/.*;base64,)(.*)" s)
      first
      (nth 2)))

(defn parse-location [location]
  (->> location
       (re-matches #".*/(.*?)\.md.*#/(.*)$")
       (drop 1)
       (str/join "-")))

(defn location-file [root dir location]
  (assert (and (seq root) dir location))
  (str root "/"
       dir "/"
       "canvas" "-"
       (parse-location location) "-"
       (str/replace (Date.) #"[: ]" "-")
       ".png"))

(defn get-bytes [s]
  (.getBytes s StandardCharsets/UTF_8))

(defn decode-canvas
  [root dir {{:keys [img-base64 location]} :params}]
  (let [file (location-file root dir location)]
    (io/make-parents file)
    (log/info ::save file)
    (try
      (with-open
        [in (io/input-stream (get-bytes (chop-header img-base64)))
         out (io/output-stream file)]
        (b64/decoding-transfer in out))
      (catch Throwable e
        (log/error e)))))

(defmethod ig/init-key
  :mathdoc.endpoint/postcanvas
  [_ {:keys [root canvas-dir logger] :as config}]
  (duct/log logger :info :mathdoc.endpoint/postcanvas [root canvas-dir])
  (routes
   (POST "/__save-canvas" req
         (do
           (duct/log logger :info ::post)
           #_(reset! state req)
           (decode-canvas root canvas-dir req)
           "success"))
   (GET "/__save-canvas" []
        (ring.util.response/response "test"))))

;;; tests

(comment

  ((ig/init-key
   :mathdoc.endpoint/postcanvas
   {:root "../../talks" :canvas-dir "canvas-save"})
   {:request-method :post, :uri "/__save-canvas", :query-string nil})

  ((:duct.core.web/handler integrant.repl/system)
   {:root "../../talks" :canvas-dir "canvas-save"})

  )

#_((-> reloaded.repl/system :postcanvas :routes) (ring.mock.request/request :post "/__save-canvas" {"location" "test"}))

#_(-> state deref :params :location)

#_(parse-location "http://localhost:3001/html-slides/lecture1.md.html?nofragments#/section-sparsest-cut-problem")

#_(location-file "../soslectures" "" "http://localhost:3001/html-slides/lecture1.md.html?nofragments#/section-sparsest-cut-problem")

#_(str (Date.))

#_(-> state deref :params keys)

#_ (-> state deref :params :img-base64)

#_ (-> state deref :params :img-base64 chop-header)
