(ns mathdoc.process.commit
  (:require [clojure.core.async :as a]
            [mathdoc.specs.process :as msp]
            [clojure.java
             [io :as io]
             [shell :as sh]]
            [clojure.spec :as s]
            [taoensso.timbre :as log]
            [integrant.core :as ig]
            [duct.core :as duct]))

;;; specs

(s/fdef process
        :args (s/cat
               :data (s/keys
                      :req-un [::msp/commitscript
                               ::msp/output
                               ::msp/targetfile
                               ::msp/root]))
        :ret keyword?)


;;; implementation

(defn relative-path
  [parent child]
  (.toString
   (.relativize
    (.toURI (io/file parent))
    (.toURI (io/file child)))))

(defn log-res [logger {:keys [out err exit] :as res}]
  (and (seq out) (duct/log logger :info out))
  (and (seq err) (duct/log logger :error err))
  res)

(defn process
  [{:keys [commitscript logger root output targetfile notefile]}]
  (assert
   (and commitscript root)
   "need commit script and root")
  (duct/log logger :info :process)
  (when (= output :success)
    ;; commit generated file
    (a/go
      (log-res logger
       (try
         (sh/sh "bash" commitscript
                (relative-path root targetfile)
                (relative-path root notefile)
                :dir (.getCanonicalPath (io/file root)))
         (catch Exception e
           (log/spy :error "commit script failed" e))))))
  output)

;;; integrant

(defmethod ig/init-key
  :mathdoc.process/commit
  [_ v]
  (process v))

;;; tests

#_(-> reloaded.repl/system :commit data/get-data)

#_ (relative-path "../soslectures/" "../soslectures/content/test.html")
