(ns mathdoc.process
  (:require [clojure.spec :as s]
            [taoensso.tufte :as tufte]
            [integrant.core :as ig]
            [taoensso.timbre :as log]
            [duct.core :as duct]
            [clojure.java.io :as io]
            [meta-merge.core :as m]
            [mathdoc.core :as core]
            [medley.core :as med]
            [duct.core :as duct]))



;; specs

(s/def ::file string?)
(s/def ::root string?)
(s/def ::content-dir string?)
(s/def ::prepend-files (s/coll-of string?))
(s/def ::append-files (s/coll-of string?))
(s/def ::target-dir string?)

(s/def ::inputfiles
  (s/keys :req-un [::file
                   ::content-dir
                   ::root
                   ::prepend-files
                   ::append-files]))

(s/def ::targetfile
  (s/keys :req-un [::file
                   ::root
                   ::target-dir]))



;; implementation

(defmethod ig/init-key
  ::root
  [_ v]
  (s/assert ::root v)
  v)

(defmethod ig/init-key
  ::file
  [_ v]
  (s/assert ::file v)
  v)

(defmethod ig/init-key
  ::inputfiles
  [_ {:keys [file
             content-dir
             root
             prepend-files
             append-files]
      :as opts}]
  (s/assert ::inputfiles opts)
  (as-> file _
    (str content-dir "/" _)
    (concat prepend-files [_] append-files)
    (map (partial str root "/") _)))

(defmethod ig/init-key
  :mathdoc.process/targetfile
  [_ {:keys [file
             root
             target-dir]
      :as opts}]
  {:pre [(s/assert ::targetfile opts)]
   :post [(s/assert string? %)]}
  (str root "/" target-dir "/" file ".html"))



(defn process-run
  [config file]
  {:pre [(s/assert map? config) (s/assert string? file)]}
  (->
   (m/meta-merge
   config
   {:mathdoc.process/file file})
   (duct/prep)
   (core/profile-init)
   :mathdoc.process/outputfile))

(defn set-namespace
  [ns k]
  {:pre [(s/assert string? ns) (s/assert keyword? k)]}
  (keyword ns (name k)))

#_(set-namespace "mathdoc.process.options" :test)

(s/def ::options map?)
(s/def ::environment #{:production :development})

(s/def :mathdoc/process
  (s/keys
   :opt-un [::environment ::core/profile? ::options]
   :req-un [::root]))

(defn get-process-config
  [{:keys [options root environment] :as config}]
  {:pre [(s/assert :mathdoc/process config)]
   :post [(s/assert map? %)]}
  (m/meta-merge
   (duct/read-config
    (io/resource "mathdoc/process/config.edn"))
   (med/map-keys (partial set-namespace "mathdoc.process.options") options)
   #_(when-not (nil? profile?)
     {::core/profile? profile?})
   (when environment
     {:duct.core/environment environment})
   {:mathdoc.process/root root}))

(defmethod ig/init-key
  :mathdoc/process
  [_ {:keys [logger profile?] :as config}]
  {:pre [(s/assert :mathdoc/process config) logger]
   :post [(s/assert fn? %)]}
  (let [process-config (get-process-config config)]
    (fn process-files [files]
      {:pre [(s/assert (s/coll-of string?) files)]}
      (duct/log logger :info :process-files (vec files))
      (tufte/add-basic-println-handler! {})
      (tufte/profile
       {:when profile?}
       (->> files
            (map #(process-run process-config %))
            doall)))))

(comment
  (ig/init-key :mathdoc/process
               {:root    "test-root"
                :profile? false
                :environment :development
                :options {:content-dir "test"}})

  (ig/init-key :mathdoc/process
               {:root    "../../talks"
                :profile? false
                :environment :development
                :options {:content-dir "content-slides"
                          :target-dir    "html-slides"
                          :include-dir   "html-slides"
                          :prepend-files ["content-slides/latexmacros.tex"]
                          :append-files  ["content-slides/metadata.yaml"]
                          :template      "html-slides/template.html"
                          :commitscript  "scripts/commit-generated.sh"}})

  )
