(ns talk.complexity
  (:require [cuerdas.core :as str]
            [garden.color :as gc]
            [garden.core :as garden]
            [garden.stylesheet :as gs]
            [reagent.core :as r]
            [talk.landscape :as landscape]))

(def color
  {:yellow  "#b58900"
   :orange  "#cb4b16"
   :color/base03  "#002b36"
   :color/base02  "#073642"
   :color/base01  "#586e75"
   :color/base00  "#657b83"
   :color/base0   "#839496"
   :color/base1   "#93a1a1"
   :color/base2   "#eee8d5"
   :color/base3   "#fdf6e3"
   :red     "#dc322f"
   :magenta "#d33682"
   :violet  "#6c71c4"
   :blue    "#268bd2"
   :cyan    "#2aa198"
   :white   "#ffffff"
   :green   "#859900"})

(def landscape-d
  {:easy-unknown (get landscape/d 0)
   :hard-unknown (get landscape/d 1)
   :easy-known (get landscape/d 2)
   :hard-known (get landscape/d 3)
   :theory (get landscape/d 4)})

#_(:easy-known landscape-d)

(def frags
  {:pvsnp 0
   :tractable 1
   :intractable 2
   :fragmented 3
   :principle 4
   :hope-unified 4
   :unknown 3
   :theory 4
   :shrink 5
   :complete 10})

(defn add-meta-key [children]
  (->>
   children
   (map-indexed (fn [i x]
                  (if (vector? x)
                    (vary-meta x assoc :key i)
                    x)))))

(defn frag-style [idx style & children]
  [:g {:style style
       :class (if idx "fragment" "")
       :data-fragment-index (or (frags idx) idx)}
   (add-meta-key children)])

(defn frag [idx & children]
  (apply frag-style idx {} children))

(def long-transition {:transition-duration "1.5s"})

(defn landscape []
  (let [{:keys [easy-known easy-unknown theory hard-known hard-unknown]} landscape-d]
    [:g {:style {:transform "translate(0px,100px)"}}
     [:g {:transform "scale(2.43) translate(-103,-403)"}
      [:path.landscape {:d (str/join " " easy-known)}]
      [:path.landscape {:d (str/join " " easy-unknown)}]
      [:path.landscape {:d (str/join " " hard-known)}]
      [:path.landscape {:d (str/join " " hard-unknown)}]
      [frag :theory
       [:path.landscape.highlight {:d (str/join " " easy-unknown)}]
       [:path.landscape.highlight {:d (str/join " " hard-unknown)}]]
      [frag-style :tractable (assoc long-transition :transition-delay "1s")
       [:path.landscape {:style {:fill (:green color)}
                         :d     (str/join " " easy-known)}]]
      [frag-style :intractable long-transition
       [:path.landscape {:style {:fill (:red color)}
                         :d     (str/join " " hard-known)}]]
      [:path.theory.fragment {:style {:stroke           (:magenta color)
                                      :fill             "none"
                                      :stroke-width     "2"
                                      #_#_:stroke-dasharray "1 1"}
                              :data-fragment-index (:theory frags)
                              :d     (str/join " " theory)}]
      [:g.fragment {:data-fragment-index (:complete frags)
                    :style {:transition-duration "1.5s"
                            :transition-delay "0.5s"}}
       [:path.landscape {:style {:fill (:green color)}
                         :d     (str/join " " easy-unknown)}]
       [:path.landscape {:style {:fill (:red color)}
                         :d     (str/join " " hard-unknown)}]]]]))

(defn fragment-style []
  (list
   [:&.complexitylandscape
   {:opacity 1
    :transition-duration "0.7s"
    :visibility "visible"
    :transform-origin "right top"
    :transform "scaleX(1.0) scaleY(1.0) translate(0px,0px)"}
   [:&.visible
    {:transform "scaleX(0.6) scaleY(0.6) translate(159px,-127px)"}]]
   [:&.globallandscape
    {:opacity 1
     :transition-duration "1s"
     :visibility "visible"
     :transform "scaleX(0.6) scaleY(0.6) translate(500px,-110px)"}
    [:&.visible
     {:transform "scaleX(1.0) scaleY(1.0) translate(0px,0px)"}]])
 )

(defn style []
  [:style
   (garden/css
    [:.reveal
     [:path.landscape
      {:stroke (:violet color)
       :fill (:cyan color)
       :stroke-width 0}
      [:&.highlight
       {:fill (gc/lighten (:cyan color) 5)}]]
     [:path.theory.fragment.visible
      {:animation "dash 2s linear forwards"}]
     [:path.theory.fragment
      {:stroke-dashoffset 500
       :stroke-dasharray 500}]
     [:.slides
      [:section
       [:.fragment
        (fragment-style)]]]])])

(defn maplabel [x y & children]
  [:div.center {:style {:width "100%"
                        :position "absolute"
                        :top "300px"
                        :transform (str "translate(" x "px, " y "px)")
                        :z-index 10000}}
   (add-meta-key children)])

(defn text [x y & children]
  [:div {:style {:position "absolute"
                 :top (str y "px")
                 :left (str x "px")
                 :width "100%"
                 :z-index 10000}}
   (add-meta-key children)])

(defn complexitylandscape []
  [:div.globallandscape.fragment
   {:style               {#_#_:transform "scaleX(0.6) scaleY(0.6) translate(500px,-110px)"}
    :data-fragment-index (frags :tractable)}
   [:div.fragment {:style               {:position  "absolute"
                                         :top       "510px"
                                         :left      "-280px"
                                         :transform "scale(2)"}
                   :data-fragment-index (frags :pvsnp)}
    [:strong.big.brighter.fragment.removevisible.slow-transition
     {:data-fragment-index (frags :tractable)}
     "P vs NP"]
    [:svg.fragment.removevisible.slow-transition
     {:viewBox             "0 0 600 900"
      :data-fragment-index (frags :tractable)
      :style                {:position "absolute"
                             :top      "40px"
                             :left     "100px"}}
     [:path
      {:d         "m372.008636,144.47113l-170.513443,68.319702l81.899216,2.074066l-129.945923,93.66095l55.039413,0.744354l-91.199234,73.070801l220.362488,-89.130707l-82.028198,0.627808l166.042419,-96.159149l-89.815613,-0.160126l147.570465,-99.986626l-107.411591,46.938927l0,0z"
       :fill      "yellow"
       :transform "rotate(-26.76797866821289 298.35443115234364,239.93659973144523)"}]]]
   [:svg.fragment.fade-out
    {:viewBox             "-200 -150 400 300"
     :data-fragment-index (frags :tractable)

     :style {#_#_:background      "rgba(255,255,255,0.1)"
             :position            "absolute"
             :transition-duration "2s"
             :transition-delay    "1s"
             :top                 "600px"
             :left                "-1000px"}}
    [:g
     {:transform "scale(1.3,0.85)"}
     [:circle {:r    150
               :cx   0
               :cy   0
               :fill (color :cyan)}]]
    [:g
     {:transform "translate(0,-12)"}
     [:text
      {:text-anchor        "middle"
       :alignment-baseline "middle"
       :fill               "white"
       :font-weight        "bold"
       :font-size          "19px"}
      "algorithms"]]
    [:g
     {:transform "translate(0,12)"}
     [:text
      {:text-anchor        "middle"
       :alignment-baseline "middle"
       :fill               (:color/base2 color)
       :font-size          "17px"}
      "we can use"]]]
   [:div.complexitylandscape.fragment
    {:data-fragment-index (:shrink frags)}
    [:div {:style {:position "absolute"
                   :top      "0px"
                   :width    "100%"
                   :left     "0px"}}
    [style]
     [:svg {:viewBox "0 0 1000 800"
            :style   {:width           "100%"
                      :height          "800px"
                      :backgroundColor "rgba(255,255,255,0)"}}
     [landscape]]]
   [text 0 30
    [:div.fragment.remove
     {:data-fragment-index (:shrink frags)}
     [frag :fragmented
      [:span.bold.yellow "complexity landscape: "]
      "our current picture is "
      [:span.magenta.italic.bold "fragmented"]]]]
   [text 0 600
    [:div.fragment.remove
     {:data-fragment-index (:shrink frags)}
     [frag :principle
      [:div.blockquote.background-shaded
       {:style {:padding "20px 0"}}
       [:div.center.italic.bold.big {:style {:font-size "110%"}}
        [:span.yellow "goal:"]
        " identify "
        [:span.magenta
         "underlying principles"]
        " that make non-convex or discrete optimization problems "
        [:span.magenta
         "(in)tractable"]]
       #_[frag :hope-unified
          [:div.center.yellow.italic.bold.big
           "is there hope for a "
           [:span.brighter "unified theory?"]
           [:br]
           " how could it look like?"]]]]]]
   [maplabel 0 10
    [:div {:style {:font-size       "120%"
                   #_#_:text-shadow (str/join " " ["3px" "0px"  (:cyan color) ","
                                                   "0px" "3px"  (:cyan color) ","
                                                   "-3px" "0px" (:cyan color) ","
                                                   "0px" "-3px" (:cyan color)])}}
     [:span.bold {:style {:color "white"}} "problems"]
     [:br]
     [:span "we wish to solve"]]]
   [frag :tractable
    [maplabel -300 170
     [:div {:style {:font-size "120%"}}
      [:span.bold.brighter "tractable"]]]]
   [frag :intractable
    [maplabel 300 -50
     [:div {:style {:font-size "120%"}}
      [:span.bold.brighter "intractable"]]]]
    [:div.fragment.remove {:data-fragment-index (:complete frags)}
     (let [q [:div {:style {:font-size "170%"}}
              [:span.magenta.bold.italic "?"]]]
       [frag :unknown
        [maplabel -300 0 q]
        [maplabel 300 130 q]
        [maplabel 30 -90 q]
        [maplabel -20 160 q]])]]])

(defn init []
  (some->> "complexitylandscape"
           (.getElementById js/document)
           (r/render [complexitylandscape])))

#_(init)

