(ns talk.datavstime
  (:require [cuerdas.core :as str]
            [garden.core :as garden]
            [garden.stylesheet :as gs]
            [mathdoc.cljs.svg :as svg]
            [reagent.core :as r]))

;;; helpers

(defn add-meta-key [children]
  (->>
   children
   (map-indexed (fn [i x]
                  (vary-meta x assoc :key i)))))

;;; config

(def color
  {:yellow  "#b58900"
   :orange  "#cb4b16"
   :red     "#dc322f"
   :magenta "#d33682"
   :violet  "#6c71c4"
   :blue    "#268bd2"
   :cyan    "#2aa198"
   :green   "#859900"})

(def data-pos
  {:d3 60
   :d4 200
   :d5 340
   :qpoly 480})

(def time-pos
  {:d3 60
   :d4 160
   :d5 260
   :poly 460
   :qpoly 660
   :exp 860})

(def alg-font-size "50px")

(def frags
  {:minmom 1
   :bruteforce 2
   :folklore 3
   :jennrich 4
   :foobi 5
   :bks 6
   :mss 7
   :hsss 8})

(defn frag [idx & children]
  [:g.fragment {:data-fragment-index (or (frags idx) idx)}
   (add-meta-key children)])

(def fast-time-pos
  (apply +
         (map *
              [0.7 0.3]
              [(:d3 time-pos) (:d4 time-pos)])))

;;; implementation

(defn defs []
  [:defs
   [:marker#triangle-marker
    {:viewBox      "0 0 10 10"
     :refX         "14"
     :refY         "5"
     :stroke       "none"
     :markerUnits  "strokeWidth"
     :markerWidth  "6"
     :markerHeight "6"
     :orient       "auto"}
    [:path {:d "M 0 0 L 10 5 L 0 10 z"}]]
   [:marker#axis-marker
    {:viewBox      "0 0 10 10"
     :refX         "6"
     :refY         "5"
     :stroke       "none"
     :fill (:yellow color)
     :markerUnits  "strokeWidth"
     :markerWidth  "6"
     :markerHeight "6"
     :orient       "auto"}
    [:path {:d "M 0 0 L 10 5 L 0 10 z"}]]])

(defn style []
  [:style
   (garden/css
    [:text.tick {:transform "scaleY(-1)"
                 :fill (:red color)
                 :font-weight "bold"
                 :font-size "50px"}]
    [:text.tick-outline
     {:transform "scaleY(-1)"
      :font-weight "bold"
      :stroke "#000000"
      :stroke-width "10px"
      :stroke-linecap "butt"
      :stroke-linejoin "miter"
      :stroke-opacity 1
      :font-size "50px"}]
    [:path.dream {:opacity 0.7
                  :stroke-width 4
                  :stroke (:magenta color)}]
    [:path.hline {:stroke-width 4
                  :stroke (:yellow color)}]
    [:text.label {:transform "scaleY(-1)"
                  :fill (:cyan color)
                  :font-weight "bold"
                  :font-size "40px"}]
    [:text.label-outline
     {:transform "scaleY(-1)"
      :fill (:cyan color)
      :font-weight "bold"
      :stroke "#000000"
      :stroke-width "10px"
      :stroke-linecap "butt"
      :stroke-linejoin "miter"
      :stroke-opacity 1
      :font-size "40px"}]
    [:tspan.super {:font-size "60%"}]
    [:text.axis {:transform "scaleY(-1)"
                 :fill (:yellow color)
                 :font-weight "bold"
                 :font-size "50px"}]
    [:circle.tick {:fill (:red color)}]
    [:path.axis {:marker-end   "url(#axis-marker)"
                 :stroke-width 6
                 :fill         "none"
                 :stroke       (:yellow color)}]
    [:.dream.fragment.visible
     {:animation "dash 1s linear forwards"}]
    [:.dream.fragment
     {:stroke-dashoffset 500
      :stroke-dasharray 500}]
    (gs/at-keyframes
     "dash"
     [:to
      {:stroke-dashoffset 0}]))
   ])

(defn gtranslate [x y & children]
  [:g {:style {:transform (str/format "translate(%spx,%spx)" x y)}}
   (add-meta-key children)])

#_(gtranslate 10 10)

(defn ylabel [& ts]
  [:g {:style {:transform "translate(-245px,-15px)"}}
   [:text.label (add-meta-key ts)]])

(defn ytick [& ts]
  [:g
   [:circle.tick {:r 14 :cx 0 :cy 0}]
   [:g {:style {:transform "translate(-20px,-15px)"}}
    [:text.tick {:textAnchor "end"} (add-meta-key ts)]]])

(defn xtick [& ts]
  [:g
   [:circle.tick {:r 14 :cx 0 :cy 0}]
   [:g {:style {:transform "translate(-30px,-70px)"}}
    [:text.tick (add-meta-key ts)]]])

(defn xtick-small [& ts]
  [:g
   [:g {:style {:transform "translate(-30px,10px)"}}
    [:text.tick-outline (add-meta-key ts)]
    [:text.tick (add-meta-key ts)]]
   [:circle.tick {:r 7 :cx 0 :cy 0}]])

(defn super [t]
  [:tspan.super {:baselineShift "super"}
   t])

(defn axis []
  [:g
   [:path.axis {:d (svg/path-d :M 0 100 :L 0 0 :L 950 0)}]
   [:path.axis {:d (svg/path-d :M 100 0 :L 0 0 :L 0 550)}]
   [:g {:style {:transform "translate(-45px,575px)"}}
    [:text.axis "data"]]
   [:g {:style {:transform "translate(970px,-15px)"}}
    [:text.axis  "time"]]
   #_data
   [:g
    [:g.fragment {:data-fragment-index 1}
     [gtranslate 0 (:d3 data-pos)
      [ylabel [:tspan "3rd mom."]]
      [ytick [:tspan "d"
              [super 3]]]]]
    [frag :foobi
     [gtranslate 0 (:d4 data-pos)
      [ylabel [:tspan "4th mom."]]
      [ytick [:tspan "d"
              [super 4]]]]]
    [frag :jennrich
     [gtranslate 0 (:d5 data-pos)
      [ylabel [:tspan "5th mom."]]
      [ytick [:tspan "d"
              [super 5]]]]]
    [frag :folklore
     [gtranslate 0 (:qpoly data-pos)
      [ylabel
       [:tspan {:x "0" :dy "-0.6em"} "log(d)"]
       [:tspan {:x "0" :dy "1.2em"} "mom."]]
      [ytick [:tspan "d"
              [super "log(d)"]]]]]
    #_time
    [frag :minmom
     [gtranslate (:d3 time-pos) 0
      [xtick [:tspan "d"
              [super 3]]]]]
    [frag :hsss
     [gtranslate fast-time-pos 0
      [xtick-small [:tspan {:style {:font-size "70%"}}
                    "3.373"]]]]
    [frag :hsss
     [gtranslate  (:d4 time-pos) 0
      [xtick [:tspan "d"
              [super 4]]]]]
    [frag :jennrich
     [gtranslate  (:d5 time-pos) 0
      [xtick [:tspan "d"
              [super 5]]]]]
    [frag :mss
     [gtranslate (:poly time-pos) 0
      [xtick [:tspan "d"
              [super "O(1)"]]]]]
    [frag :folklore
     [gtranslate (:qpoly time-pos) 0
      [xtick [:tspan "d"
              [super "log(d)"]]]]]
    [frag :bruteforce
     [gtranslate (:exp time-pos) 0
      [xtick [:tspan "2"
              [super "d"]]]]]]])

(defn classical-alg [& ts]
  [:g [gtranslate 0 40
       [:text.label-outline {:textAnchor "middle"
                             :style {:font-size alg-font-size}}
        (add-meta-key ts)]
       [:text.label {:textAnchor "middle"
                     :style {:font-size alg-font-size}}
        (add-meta-key ts)]]
   [:circle {:cx 0 :cy 0 :r 20
             :style {:fill (:cyan color)}}]])

(defn dream-alg [& ts]
  [:g [gtranslate 0 40
       [:text.label-outline {:textAnchor "middle"
                             :style {:fill (:magenta color)
                                     :font-size alg-font-size}}
        (add-meta-key ts)]
       [:text.label {:textAnchor "middle"
                     :style {:fill (:magenta color)
                             :font-size alg-font-size}}
        (add-meta-key ts)]]
   [:circle {:cx 0 :cy 0 :r 20
             :style {:fill (:magenta color)}}]])

(defn figure []
  [:div
   #_[:div [:div.fragment "hello"]
      [:div.fragment "how are you"]
      [:div.fragment "bye"]]
   [:svg {:style {:backgroundColor "rgba(255,255,255,0)"
                  :transform "scaleY(-1)"
                  :width "1030px"
                  :height "600px"}
          :viewBox "-250 0 1360 500"}
    [defs]
    [style]

    [frag :hsss
     [:path.dream.fragment
      {:d (svg/path-d :M (:d5 time-pos) (:d5 data-pos)
                      :L fast-time-pos (:d3 data-pos))
       :data-fragment-index (:hsss frags)}]]

    [frag :mss
     [:path.dream.fragment
      {:d (svg/path-d :M (:d5 time-pos) (:d5 data-pos)
                      :L (:poly time-pos) (:d3 data-pos))
       :data-fragment-index (:mss frags)}]]

    [frag :bks
     [:path.dream.fragment
      {:d (svg/path-d :M (:qpoly time-pos) (:qpoly data-pos)
                      :L (:qpoly time-pos) (:d3 data-pos))
       :data-fragment-index (:bks frags)}]]

    [frag :minmom
     [gtranslate 0 60
      [gtranslate 930 -10 [:text.axis {:style {:font-size "40px"}} "min mom."]]
      [:path.hline {:style {:opacity 0.7}
                    :d (svg/path-d :M 0 0 :L 925 0)}]]]

    [axis]
    [frag :foobi
     [gtranslate (:d5 time-pos) (:d4 data-pos)
      [classical-alg [:tspan "FOOBI'07"]]]]
    [frag :bruteforce
     [gtranslate (:exp time-pos) (:d3 data-pos)
      [classical-alg [:tspan "brute force"]]]]
    [frag :folklore
     [gtranslate (:qpoly time-pos) (:qpoly data-pos)
      [classical-alg [:tspan "folklore"]]]]
    [frag :jennrich
     [gtranslate (:d5 time-pos) (:d5 data-pos)
      [classical-alg [:tspan "Jennrich'70"]]]]
    [frag :bks
     [gtranslate (:qpoly time-pos) (:d3 data-pos)
      [dream-alg [:tspan "BKS'15"]]]]
    [frag :mss
     [gtranslate (:poly time-pos) (:d3 data-pos)
      [dream-alg [:tspan "MSS'16"]]]]
    [frag :hsss
     [gtranslate
      fast-time-pos (:d3 data-pos)
      [dream-alg [:tspan "HSSS'16"]]]]]])

(defn mount-root []
  (some->> "datavstime"
           (.getElementById js/document)
           (r/render [figure])))

(defn init []
  #_(println "data vs time")

  (mount-root))
