(ns talk.simplification
  (:require [cuerdas.core :as str]
            [garden.core :as garden]
            [garden.stylesheet :as gs]
            [mathdoc.cljs.svg :as svg]
            [reagent.core :as r]
            [garden.color :as gc]
            [clojure.walk :as walk]
            [cuerdas.core :as str]
            [mathdoc.cljs.svg :as svg]))

;;; context

(def context
  (let [color {:color/yellow  "#b58900"
               :color/orange  "#cb4b16"
               :color/red     "#dc322f"
               :color/magenta "#d33682"
               :color/violet  "#6c71c4"
               :color/blue    "#268bd2"
               :color/cyan    "#2aa198"
               :color/green   "#859900"
               :color/base03  "#002b36"
               :color/base02  "#073642"
               :color/base01  "#586e75"
               :color/base00  "#657b83"
               :color/base0   "#839496"
               :color/base1   "#93a1a1"
               :color/base2   "#eee8d5"
               :color/base3   "#fdf6e3"}]
    (merge
     color
     {:cycle/radius   "25"
      :font/size      "10px"
      :font/color     (:color/base1 color)
      :font/bigsize   "15px"
      :font/largesize "14px"

      :quantum/color (:color/orange color)

      :frag/general-alg       0
      :frag/simplify          1
      :frag/simple-alg        2
      :frag/lp                3
      :frag/physical          4
      :frag/entropy           5
      :frag/equations         6
      :frag/maxentropy        7
      :frag/exponentialfamily 8
      :frag/sosapprox         9
      :frag/quantumsdp        10
      :frag/quantumrepresent  11}
     #_(zipmap
        [:frag/general-alg
         :frag/simplify
         :frag/simple-alg
         :frag/lp
         :frag/physical
         :frag/entropy
         :frag/equations
         :frag/maxentropy
         :frag/exponentialfamily
         :frag/sosapprox
         :frag/quantumsdp
         :frag/quantumrepresent] (repeat 0)))))

;;; refs

(defrecord Ref [key])

(defn ref [k]
  (->Ref k))

(defn ref? [x]
  (instance? Ref x))

(defn expand
  [form]
  (walk/postwalk
   (fn [x]
     (if (ref? x)
       (context (:key x))
       x))
   form))

(def transform-arrow
  [:g
   {:transform "translate(0,-16.2)"}
   [:path {:d "M31 16l-15-15v9h-16v12h16v9z"
           :transform "scale(3.0,1.0)"
           :fill (ref :color/cyan)}]])

(def physical-representation
  [:g
   [:g.fragment
    {:data-fragment-index (ref :frag/physical)}
    [:text
     {:font-size          (ref :font/size)
      :alignment-baseline "hanging"
      :dy                 0
      :font-style         "italic"
      :font-weight        "bold"
      :fill               (ref :font/color)}
     "physical representations:"]
    [:g.fragment.removevisible
     {:data-fragment-index (ref :frag/quantumrepresent)}
     [:g
      {:transform "translate(0,12)"}
      [:text
       {:font-size          (ref :font/size)
        :alignment-baseline "hanging"
        :fill               (ref :color/magenta)}
       [:tspan
        {:alignment-baseline "hanging"}
        "distributions over "]
       [:tspan
        {:fill               (ref :color/blue)
         :alignment-baseline "hanging"}
        "n"]
       [:tspan
        {:alignment-baseline "hanging"}
        " bits"]]]]
    [:g.fragment.transition-delay.fade-right
     {:data-fragment-index (ref :frag/quantumrepresent)}
     [:g
      {:transform "translate(0,12)"}
      [:text
       {:font-size          (ref :font/size)
        :alignment-baseline "hanging"
        :fill               (ref :quantum/color)}
       [:tspan
        {:alignment-baseline "hanging"}
        "quantum state of "]
       [:tspan
        {:fill               (ref :color/blue)
         :alignment-baseline "hanging"}
        "n"]
       [:tspan
        {:alignment-baseline "hanging"}
        " qubits"]]]]]

   [:g.fragment
    {:data-fragment-index (ref :frag/entropy)}
    [:text
     {:font-size          (ref :font/size)
      :alignment-baseline "hanging"
      :dy                 "2.4em"
      :fill               (ref :color/base2)}
     "entropy deficit "
     [:tspan
      {:fill               (ref :color/blue)
       :alignment-baseline "hanging"}
      "O(log n)"]]]

   [:g.fragment.fade-right
    {:data-fragment-index (ref :frag/quantumsdp)}
    [:g
     {:transform "translate(0,80)"}
     [:text
      {:font-size          (ref :font/largesize)
       :alignment-baseline "hanging"
       :fill               (ref :color/base1)}
      [:tspan
       {:fill               (ref :color/yellow)
        :alignment-baseline "hanging"
        :font-weight        "bold"}
       "simplify SDP algorithms:"]
      " same"]
     [:text
      {:dy                 "1.2em"
       :font-size          (ref :font/largesize)
       :alignment-baseline "hanging"
       :fill               (ref :color/base1) }
      " but in"
      [:tspan
       {:font-weight        "bold"
        :font-style         "italic"
        :font-size          (ref :font/largesize)
        :fill               (ref :quantum/color)
        :alignment-baseline "hanging"}
       " quantum physics"]]
     [:g
      {:style
       {:transform "translate(0px,-42px)"}}
      [:image {:href  "/figure/quantum.png"
                    :style {:width  "40px"
                  
                            :height "40px"}}]]]]])

(def algorithms
  [:g.fragment
   {:data-fragment-index (ref :frag/general-alg)}
   [:text
    {:font-size (ref :font/size)
     :fill (ref :color/base2)
     :dy "-2.4em"}
    "poly-size"]
   [:text
    {:font-size (ref :font/size)
     :fill (ref :color/magenta)
     :dy "-1.2em"}
    "LP "
    [:tspan.fragment.remove-font
     {:data-fragment-index (ref :frag/lp)}
     " / SDP"]
    [:tspan.fragment.appear-font
     {:font-weight "bold"
      :data-fragment-index (ref :frag/quantumsdp)
      :font-style "italic"}
     " / SDP"]
    " algorithms"]
   [:text
    {:font-size (ref :font/size)
     :fill (ref :color/magenta)}
    "for "
    [:tspan {:fill (ref :color/blue)}
     "n"]
    " Boolean vars"]])

(defn mjsvgtypeset [x]
  (when js/MathJax
    (js/MathJax.Hub.Queue
     ;; #js ["setRenderer" js/MathJax.Hub "SVG"]
     #js ["Reprocess" js/MathJax.Hub x]
     #js ["Typeset" js/MathJax.Hub x]
     ;;#js ["setRenderer" js/MathJax.Hub "HTML-CSS"]
)))

(defn equations []
  (expand
    (let [attr {:ref (fn [x]
                       (print "mj typeset" x)
                       (mjsvgtypeset x))
                :style {:position "absolute"
                        :left "440px"
                        :top "525px"}}]
      [:div
      [:div
       attr
       [:div.fragment.removevisible
        {:data-fragment-index (ref :frag/quantumrepresent)}
        [:div.fragment
         {:data-fragment-index (ref :frag/equations)}
         [:p
          "$\\mathbb E_p f_i = \\mathbb E_{p'} f_i$"]]]]
      [:div
       attr
       [:div.fragment.fade-right.transition-delay
        {:data-fragment-index (ref :frag/quantumrepresent)}
        [:p
         (str/format
          "$\\color{%s}{\\mathrm{Tr}(\\rho F_i) = \\mathrm{Tr}(\\rho' F_i)}$"
          (:color/orange context))]]]])))

(defn exponential []
  (expand
   [:div
    [:div
     {:ref (fn [x]
             (print "mj typeset" x)
             (mjsvgtypeset x))
      :style {:position "absolute"
              :text-align "right"
              :left "550px"
              :width "610px"
              :top "565px"}}
     [:div.fragment.removevisible.transition-delay
      {:data-fragment-index (ref :frag/quantumrepresent)}
      [:div.fragment
       {:data-fragment-index (ref :frag/exponentialfamily)}
       [:p {:style {:margin-bottom "30px"}}
        "$p'\\propto \\exp(\\sum_i \\lambda_i\\cdot f_i)\\quad\\quad$"]]]
     [:div.fragment
      {:data-fragment-index (ref :frag/sosapprox)}
      [:div.fragment.removevisible.transition-delay
       {:data-fragment-index (ref :frag/quantumrepresent)}
       [:p "$\\exp(x)\\approx \\left(\\sum_k^{\\le O(\\log n)} \\frac 1{k!} (\\frac x 2)^k\\right)^2\\quad$"]]
      [:p "$\\exp$ almost syntactically nonnegative"
       [:span {:style {:width "50px"}}]]]]
    [:div
     {:ref (fn [x]
             (print "mj typeset" x)
             (mjsvgtypeset x))
      :style {:position "absolute"
              :text-align "right"
              :left "590px"
              :width "610px"
              :top "565px"}}
     [:div.fragment.transition-delay
      {:data-fragment-index (ref :frag/quantumrepresent)}
      [:p {:style {:margin-bottom "30px"}}
       (str/format
        "$\\color{%s}{\\rho'}\\propto \\exp(\\sum_i \\lambda_i\\cdot \\color{%s}{F_i})\\quad\\quad$"
        (:color/orange context)
        (:color/orange context))]
      [:p (str/format
           "$\\exp(\\color{%s}{X})\\approx \\left(\\sum_k^{\\le O(\\log n)} \\frac 1{k!} (\\frac {\\color{%s}{X}} 2)^k\\right)^2\\quad$"
           (:color/orange context)
           (:color/orange context))]]]]))

#_(exponential)

(defn simplification
  []
  (expand
   [:div
    [:div.big.cyan.bold
     {:style {:margin-left "-100px"}}
     "key step of unconditional lower bound"]
    [equations]
    [exponential]
    [:svg
     {:viewBox "-25 -75 355 250"
      :width   "1235px"
      :style   {#_#_:background "rgba(255,255,255,0.1)"
                :margin-left    "-150px"}}
     [:g.fragment
      {:data-fragment-index (ref :frag/physical)}
      [:path
       {:d     (svg/path-d :m -20 0 :l 1100 0)
        :style {:stroke           (ref :color/base01)
                :stroke-dasharray "3 1"
                :stroke-width     0.7}}]]
     [:g
      {:transform "translate(-10,0)"}
      (let [offset 20]
        [:g
         [:g
          {:transform (str/format "translate(0,-%s)" offset)}
          algorithms]
         [:g
          {:transform (str/format "translate(0,%s)" offset)}
          physical-representation]])
     
      [:g
       {:transform "translate(125,0)"}
       [:g.fragment
        {:data-fragment-index (ref :frag/general-alg)}
        [:circle {:cx   0
                  :cy   0
                  :r    (ref :cycle/radius)
                  :fill (ref :color/red)}]
        [:g
         {:transform "translate(-20,-20)"}
         [:image {:href  "/figure/knot.png"
                  :style {:width  "40px"
                          :height "40px"}}]]]
       (let [offset 30]
         [:g
          [:g.fragment
           {:data-fragment-index (ref :frag/general-alg)}
           [:text
            {:font-size   (ref :font/bigsize)
             :fill        (ref :color/red)
             :font-weight "bold"
             :text-anchor "middle"
             :dy          (- offset)}
            "A"]]
          [:g.fragment.removevisible.transition-delay
           {:data-fragment-index (ref :frag/quantumrepresent)}
           [:g.fragment
            {:data-fragment-index (ref :frag/physical)}
            [:text
             {:font-size          (ref :font/bigsize)
              :fill               (ref :color/red)
              :text-anchor        "middle"
              :font-weight        "bold"
              :alignment-baseline "hanging"
              :dy                 offset}
             "p"]]]
          [:g.fragment.transition-delay
           {:data-fragment-index (ref :frag/quantumrepresent)}
           [:text
            {:font-size          (ref :font/bigsize)
             :fill               (ref :color/red)
             :text-anchor        "middle"
             :font-weight        "bold"
             :alignment-baseline "hanging"
             :dy                 offset}
            "ρ"]]])]
      [:g {:transform "translate(155,0)"}
       [:g.fragment
        {:data-fragment-index (ref :frag/simplify)}
        transform-arrow
        [:text {:font-size   (ref :font/bigsize)
                :fill        (ref :color/cyan)
                :font-weight "bold"
                :font-style  "italic"
                :dy          -25}
         "simplify"]
        [:text {:font-size (ref :font/size)
                :fill      (ref :color/yellow)
                :dy        -45}
         "preserve guarantees"]]
       [:g.fragment
        {:data-fragment-index (ref :frag/maxentropy)}
        [:text {:font-size          (ref :font/bigsize)
                :fill               (ref :color/cyan)
                :font-weight        "bold"
                :font-style         "italic"
                :alignment-baseline "hanging"
                :dy                 25}
         "maximize entropy"]]
       [:g.fragment
        {:data-fragment-index (ref :frag/equations)}
        [:text {:font-size          (ref :font/size)
                :fill               (ref :color/yellow)
                :alignment-baseline "hanging"
                :dy                 45}
         "satisfy equations"]]]]
     [:g
      {:transform "translate(290,0)"}
      [:g.fragment
       {:data-fragment-index (ref :frag/simple-alg)}
       [:circle {:cx   0
                 :cy   0
                 :r    (ref :cycle/radius)
                 :fill (ref :color/green)}]
       [:g {:transform "translate(-17,-17)"}
        [:image {:href  "/figure/tidyknot.png"
                 :style {:width  "35px"
                         :height "35px"}}]]]
      (let [offset 57
            attr   {:font-size   (ref :font/size)
                    :fill        (ref :color/magenta)
                    :font-weight "bold"
                    :font-style  "italic"
                    :text-anchor "middle"
                    #_#_:dy      offset}]
        [:g
         [:g.fragment
          {:data-fragment-index (ref :frag/simple-alg)}
          [:g
           {:transform (str/format "translate(0,%s)" (- offset))}
           [:text
            attr #_(update attr :dy -)
            "captured by SOS"]]]
         [:g.fragment
          {:data-fragment-index (ref :frag/exponentialfamily)}
          [:g
           {:transform (str/format "translate(0,%s)" offset)}
           [:text
            (assoc attr :alignment-baseline "hanging")
            [:tspan.fragment.transition-delay
             {:alignment-baseline  "hanging"
              :fill (ref :quantum/color)
              :data-fragment-index (ref :frag/quantumrepresent)
              :dy -10}
             "matrix-"]]
           [:text
            (assoc attr :alignment-baseline "hanging")
            [:tspan
             {:alignment-baseline  "hanging"}
             "exponential family"]]]]])
      (let [offset 30
            attr   {:font-size   (ref :font/bigsize)
                    :fill        (ref :color/green)
                    :text-anchor "middle"
                    :font-weight "bold"
                    :dy          offset}]
        [:g
         [:g.fragment
          {:data-fragment-index (ref :frag/simple-alg)}
          [:text
           (update attr :dy -)
           "A'"]]
         [:g.fragment.removevisible.transition-delay
          {:data-fragment-index (ref :frag/quantumrepresent)}
          [:g.fragment
           {:data-fragment-index (ref :frag/maxentropy)}
           [:text
            (assoc attr :alignment-baseline "hanging")
            "p'"]]]
         [:g.fragment.transition-delay
          {:data-fragment-index (ref :frag/quantumrepresent)}
          [:text
           (assoc attr :alignment-baseline "hanging")
           "ρ'"]]])]]]))

(defn init []
  (some->> "simplification"
           (.getElementById js/document)
           (r/render [simplification])))
