(ns mathdoc.cljs.drawmenu
  (:require-macros [cljs.core.async.macros :refer [alt! go go-loop]])
  (:require
   [cljs.core.async :as a]
   [hickory.core :as h]
   [hickory.render :as hrender]
   [dommy.core :as dommy :refer-macros [sel sel1]]
   [clojure.walk :as walk]))

(def colors
  {:yellow  "#b58900"
   :orange  "#cb4b16"
   :red     "#dc322f"
   :magenta "#d33682"
   :violet  "#6c71c4"
   :blue    "#268bd2"
   :cyan    "#2aa198"
   :green   "#859900"})

#_(dommy/listen! (sel1 "#paintbrush-red")
                 :click #(aset js/Pen "colors" "fg" (:red colors)))

#_(dommy/listen! (sel1 "#paintbrush-yellow")
                 :click #(aset js/Pen "colors" "fg" (:yellow colors)))

(defn listen-id-color [id color]
  #_(.log js/console id color)
  (when-let [elem (sel1 (str "#" id))]
    (aset elem "style" "color" color)
    (dommy/listen!
     elem
     :click #(aset js/Pen "colors" "fg" color))))

#_(dommy/bounding-client-rect (sel1 "#paintbrush-yellow"))

(defn add-color-listeners []
  (doseq [[colorname color] colors]
    #_(.log js/console colorname color)
    (listen-id-color
     (str "paintbrush-" (name colorname))
     color)))

(defn clear-board []
  (.clearMemory js/Board))

(defn make-image [src]
  (doto (new js/Image)
    (aset "src" src)))

#_(make-image "lecture1-title.png")

(defn draw-image [img]
  ;; scale so that height matches
  ;; center the image horizontally
  (let [canvas (sel1 "#board")
        ratio (/ (.-height canvas) (.-height img))
        shift (* 0.5 (- (.-width canvas) (* ratio (.-width img))))]
    (->
     canvas
     (.getContext "2d")
     (.drawImage img
                 0 0 (.-width img) (.-height img)
                 shift 0 (* ratio (.-width img)) (* ratio (.-height img))))))

(defn load-image [src]
  (let [img (make-image src)]
    (-> img
        (aset "onload"
              #(do
                 (clear-board)
                 (draw-image img))))))

#_(load-image "lecture1-title.png")

#_(dommy/remove-style!
   (sel1 "#board")
   "opacity")

#_(dommy/remove-class!
   (sel1 "#board")
   "fadedout")

#_(dommy/add-class!
   (sel1 "#board")
   "fadedout")

#_(dommy/toggle-class!
   (sel1 "#board")
   "fadedout")

(defn load-elem-image [elem]
  (when-let [src (dommy/attr elem "data-canvas-src")]
    (let [board (sel1 "#board")]
      (load-image src)
      #_(go
        #_(.log js/console "load")
        #_(dommy/remove-style! board "opacity")
        #_(dommy/add-class! board "fadedout")
        #_(a/<! (a/timeout 100))
        #_(dommy/remove-class! board "fadedout")))))

(defn load-slide-image-listener [ev]
  (load-elem-image (aget ev "currentSlide")))

(defn load-slide-fragment-listener [ev]
  (load-elem-image (aget ev "fragment")))

#_(load-elem-image (sel1 "#section-optimization"))

(defn get-canvas-data []
  (.toDataURL (sel1 "#board")))

(defn get-location []
  (str (aget js/window "location")))

(defn post-canvas []
  (.log js/console "post canvas")
  (->
   js/jQuery
   (.ajax
    #js {:type "POST"
         :url "/__save-canvas"
         :data
         #js {:img-base64 (get-canvas-data)
              :location (get-location)}})
   (aget "done"))
  #_(.log js/console "posted"))

#_((post-canvas) (fn [msg] (print
                           "test" msg)))

(defn add-trash-listener []
  (dommy/listen!
   (sel1 "#trash")
   :click clear-board))

(defn add-download-listener []
  (dommy/listen!
   (sel1 "#download")
   :click post-canvas))

;; if (window.location.search.match( /nofragments/gi ))
;; { Reveal.configure({fragments: false});}


(defn maybe-hide-canvas []
  (when (-> js/window .-location .-search (.match #"nocanvas"))
    (.log js/console "hide canvas")
    (-> (sel1 "#board")
        (dommy/set-style! "display" "none"))))

(defn fix-top [slide]
  (when (dommy/has-class? slide "align-top")
    #_(print "fix-top" (.-id slide))
    (dommy/set-style!
     slide
     "top" "0px")))

#_ (-> (sel1 "#section-section-28")
       (dommy/set-style! "top" "0px"))

(defn maybe-center-slides []
  (doseq [slide (sel ".reveal section")]
    (when-not (dommy/has-class? slide "align-top")
      (dommy/add-class! slide "center"))))

#_(defn check-top-slide [ev]

  ;; doesn't work so well; better set top of slides to 0 if they have class align-top
  #_(if (dommy/has-class? (.-currentSlide ev) "align-top")
    (js/Reveal.configure #js {"center" false})
    (js/Reveal.configure #js {"center" true}))

  #_(doseq [slide ["previousSlide" "currentSlide" "nextSlide"]]
    (print "slide" slide "aget" (aget ev slide))
    (when (aget ev slide)
      (fix-top (aget ev slide))))

  #_(when (dommy/has-class? (.-currentSlide ev) "align-top")
    (dommy/set-style!
     (.-currentSlide ev)
     "top" "0px")))



(defn ^:export init []
  (.log js/console "drawmenu")
  (.addEventListener
   js/Reveal
   "slidechanged"
   (juxt
    load-slide-image-listener
    clear-board))

  (.addEventListener
   js/Reveal
   "fragmentshown"
   load-slide-fragment-listener)

  (.addEventListener
   js/Reveal
   "ready"
   (juxt
    maybe-hide-canvas
    maybe-center-slides
    add-download-listener
    add-trash-listener
    add-color-listeners)))
