(ns talk.contract
  (:require
   [cuerdas.core :as str]
   [reagent.core :as r :refer [atom]]
   [mathdoc.cljs.svg :as svg]))

(def colors
  {:yellow  "#b58900"
   :orange  "#cb4b16"
   :red     "#dc322f"
   :magenta "#d33682"
   :violet  "#6c71c4"
   :blue    "#268bd2"
   :cyan    "#2aa198"
   :green   "#859900"})

;; use fragment class to interact with reveal

(defonce state
  (atom
   {}))

(defn add-meta-key [children]
  (->>
   children
   (map-indexed (fn [i x]
                  (vary-meta x assoc :key i)))))

(defn svg [opts & children]
  (let [{:keys [padding scale]} @state]
    [:svg
     {:width "100%"
      :height "300px"
      :viewBox (str/format
                "$x $y $width $height"
                {:x      (- padding)
                 :y      (- padding)
                 :width  (+ scale (* 2 padding))
                 :height (+ scale (* 2 padding))})}
     (add-meta-key children)]))

(defn grid-pattern []
  [:pattern {:id "grid"
             :width 80
             :height 80
             :patternUnits "userSpaceOnUse"}
   [:path {:d (svg/path-d :M 80 0 :L 0 0 0 80)
           :fill "none"
           :stroke "#444"
           :stroke-width "10px"}]])

(defn solid-side [w h]
  [:div {:style {:position "absolute"
                 :font-size "0px"}}
   [:svg {:viewBox (str/format "0 0 $width $height"
                               {:width (* w 80)
                                :height (* h 80)})
          :width (str (* w 20) "px")
          :height (str (* h 20) "px")}
    [:g
     [:rect {:x      0
             :y      0
             :width  (* w 80)
             :height (* h 80)
             :fill   (:cyan colors)}]
     [:rect {:x      0
             :y      0
             :width  (* w 80)
             :height (* h 80)
             :fill   "url(#grid)"}]]]])

(defn side-transform [rotate]
  {:position "absolute"
   :font-size "0px"
   :transform (str rotate " " "translateZ(100px)")})

(defn cube []
  [:div {:style {:perspective "1000px"
                 :transform "translateX(100px)"
                 :transform-style "flat"}}
   [:div {:style {:transform-style "preserve-3d"
                  :transform "rotateX(-15deg)"}}
    [:div.front {:style (side-transform "")}
     [solid-side 10 10]]
    [:div.top {:style (side-transform "rotateX(90deg)")}
     [solid-side 10 10]]
    [:div.right {:style (side-transform "rotateY(90deg)")}
     [solid-side 10 10]]
    [:div.left {:style (side-transform "rotateY(-90deg)")}
     [solid-side 10 10]]
    [:div.bottom {:style (side-transform "rotateX(-90deg)")}
     [solid-side 10 10]]
    [:div.back {:style (side-transform "rotateY(-180deg)")}
     [solid-side 10 10]]]])

(defn div-transform [transform & children]
  #_(print transform)
  [:div {:style {:position "absolute"
                 :transform transform}}
   (add-meta-key children)])

(defn solid [a b c]
  [:div #_{:style {:position "absolute"}}
   #_ "front"
   [div-transform ""
    [solid-side a b]]
   #_ "back"
   [div-transform (str/format "translateZ(%spx)"
                              (* c -20))
    [solid-side a b]]
   #_ "top"
   [div-transform (str/format "translateZ(%spx) rotateX(90deg)"
                              (* c -20))
    [solid-side a c]]
   #_ "bottom"
   [div-transform (str/format "translateZ(%spx) translateY(%spx) rotateX(90deg)"
                              (* c -20) (* b 20))
    [solid-side a c]]
   #_ "right"
   [div-transform (str/format "translateX(%spx) translateY(0px) translateZ(-%spx) rotateY(90deg) rotateZ(90deg)"
                              (* a 20) (+ 1 (* c 20)))
    [solid-side b c]]
   #_"left"
   [div-transform (str/format "translateZ(-%spx) rotateY(90deg) rotateZ(90deg)"
                              (+ 1 (* c 20)))
    [solid-side b c]]])


(defn scene []
  [:div {:style {:perspective "1200px"
                 :transform "translateX(300px)"
                 :transform-style "flat"}}
   [:div {:style {:transform-style "preserve-3d"
                  :transform "rotateX(-15deg)"}}
    [:div {:style {:transform "translateX(10px)"
                   :position "absolute"}}
     [solid 2 5 10]]
    [solid 2 5 10]]])

(defn figure []
  [:div "hello"
   [:pre (str @state)]
   [:svg {:width 0 :height 0}
    [grid-pattern]]
   #_[solid-side 5 10]
   [scene]
   ])

(defn init-state! []
  (swap! state merge {:padding 500
                      :scale 500}))

(defn mount-root []
  (->> "contract"
       (.getElementById js/document)
       (r/render [figure])))

(defn init! []
  #_(print "init")
  (init-state!)
  (mount-root))
