(ns mathdoc.compiler.webjars
  (:require [com.stuartsierra.component :as component]
            [clojure.java.io :as io]
            [mathdoc.utils.serialize :refer [serialize]]
            [clojure.string :as str]
            [clojure.spec :as s]
            [integrant.core :as ig])
  (:import org.webjars.WebJarAssetLocator))


(def ^:private webjars-pattern
  #"META-INF/resources/webjars/([^/]+)/([^/]+)/(.*)")

(defn- asset-path [prefix resource]
  (when-let [[_ name version path] (re-matches webjars-pattern resource)]
    (str prefix "/" name "/" path)))

(defn- asset-map [^WebJarAssetLocator locator prefix]
  (->> (.listAssets locator "")
       (map (juxt (partial asset-path prefix) identity))
       (remove (comp nil? first))
       (into {})))

(defn- assets
  ([]
   (assets "/assets"))
  ([prefix]
   (asset-map (WebJarAssetLocator.) prefix)))

(defn- copy-assets [prefix destdir xf]
  (doseq [[path resource] (sequence xf (assets prefix))]
    (println "copy" path)
    (io/make-parents (str destdir path))
    (io/copy (-> resource io/resource serialize)
             (io/file (str destdir path)))))



(s/def ::remove-pattern string?)
(s/def ::destdir string?)

(s/def ::config
  (s/keys :req-un [::destdir ::remove-pattern]))



(derive :mathdoc.compiler/webjars :duct/compiler)

(defmethod ig/init-key
  :mathdoc.compiler/webjars
  [_ {:keys [remove-pattern destdir] :as config}]

  (s/assert ::config config)

  (let [asset-xf (remove (comp (partial re-matches (re-pattern remove-pattern))
                            first))]

    (println "\n**Copy static assets**")

    (->> asset-xf
         (copy-assets "/assets" destdir)
         (with-out-str)
         (str/split-lines)
         (count)
         (format "(silenced %d lines)")
         (println))

    destdir))

(comment

  (ig/init-key
   :mathdoc.compiler/webjars
   {:remove-pattern ".*/MathJax/.*\\.png$"
    :destdir "target/mathdoc/compiler/webjars/public"})

  )
