(ns latex-compile.helpers
  (:require [clojure.java.io :as io]
            [plumbing.graph :as graph]
            [text-decoration.core :as deco]))

(defn cache-filter [predicate signature]
  (fn [step]
    (let [last-received (atom nil)]
      (fn
        ([result] (step result))
        ([result input]
           (if (predicate input)
             (let [signature (signature input)]
               (if (= @last-received signature)
                 (do #_(println "cache!")
                     result)
                 (do (reset! last-received signature)
                     (step result input))))
             (step result input)))))))

;; helper for map manipulation
(defn rename-key [m k k']
  (assoc (dissoc m k) k' (get m k)))

;; helper to translate opts to args
(defn opt->arg [opt & {:keys [seps]
                       :or {seps ["-" "="]}}]
  (->> opt
       (map name)
       (interleave seps)
       (apply str)))

(defn opts->args [opts]
  (map opt->arg opts))

;; helper to translate opts to map
(defn opts->map
  [opts]
  (->> opts
       (filter (comp even? count))
       (into {})
       (#(assoc % :flags (->> opts
                             (filter (comp odd? count))
                             (apply concat))))))

;; helper to remove and update options
(defn opts-dissoc [opts k]
  (->> opts
       (remove (comp (partial = k) first))
       (vec)))

(defn opts-update [opts k v]
  (-> opts
      (opts-dissoc k)
      (conj [k v])))


;; helper to extract path from file
(defn path [f]
  (.getPath (io/file f)))

;; helpers for computation graphs
(defn graph-run [g m & {:keys [keep-key?] :or {keep-key? (constantly true)}}]
  (let [pruned-map (apply dissoc m (keys g))]
    (-> (graph/run g pruned-map)
        (merge pruned-map)
        ((partial apply dissoc) (remove keep-key? (keys g))))))

;; helpers for normalizing function outputs

(defn map-multiply [m k]
  "create list of maps based on key with list value"
  (map (partial assoc m k) (get m k)))

(defn maplist-normalize [x]
  (if (map? x) [x] x))

;; helpers for redirecting output
(defn juxt-out [f]
  (fn [& args]
    (let [s (new java.io.StringWriter)]
      [(binding [*out* s]
         (apply f args))
       (str s)])))

(defn ->relative-path [base file]
  (-> base
      io/file
      .toURI
      (.relativize (-> file io/file .toURI))
      .getPath))

(defn slurp-maybe [f]
  (when (.exists (io/file f))
    (slurp f)))

(defn printfln-deco [deco & args]
  (let [deco-map {:red deco/red
                  :bold deco/bold
                  :cyan deco/cyan
                  :yellow deco/yellow
                  :green deco/green}]
    (assert (every? (partial contains? deco-map) deco))
    (println
     ((if (= "Linux" (System/getProperty "os.name"))
        (apply comp (map deco-map deco))
        identity)
      (apply format args)))))

#_(System/getProperty "os.name")
"Linux"
