(ns latex-compile.bibtex
  (:require [clojure.java.shell :as sh]
            [latex-compile.helpers :as helpers :refer [graph-run path rename-key]]
            [me.raynes.fs :as fs]
            [plumbing.core :refer [fnk]])
  (:import (java.io File)))

(defn update-ext [f ext]
  (path (fs/file (fs/parent f) (str (fs/name f) ext))))


#_{:output-file "/home/david/p/ugcbook/tmp/latex/book/assemble.pdf"
   :file "../../book.tex"}

;; find aux file, read content of aux file, remember original file name

;; cache this file (only act if aux file changed)


#_(defn output-file->aux-file [f]
    (fs/file (fs/parent f) (str (fs/name f) ".aux")))

#_(slurp (output-file->aux-file
          "/home/david/p/ugcbook/tmp/latex/book/assemble.pdf"))

(def auxfile-graph
  {:aux-file (fnk [output-file]
                  (update-ext output-file ".aux"))
   :aux-content (fnk [aux-file]
                     (slurp aux-file))})

#_((:aux-file bibtex-graph) {:output-file "/home/david/p/ugcbook/tmp/latex/book/assemble.pdf"
                             :file "../../book.tex"})


(defn read-auxfile [{:keys [output-file file] :as m}]
  (-> (graph-run auxfile-graph m)
      (dissoc :output-file)
      (rename-key :file :source-file)
      (rename-key :aux-file :file)
      (rename-key :aux-content :content)))

#_(keys (read-auxfile {:output-file "/home/david/p/ugcbook/tmp/latex/book/assemble.pdf"
                       :file "../../book.tex"}))

#_(dissoc (read-auxfile {:output-file "/home/david/p/ugcbook/tmp/latex/book/assemble.pdf"
                       :file "../../book.tex"}) :content)

;; if biblatex: copy blx to parent directory of original file (check for conflict?)
;; run bibtex8 in parent directory of original file on aux file
;; find produced bbl file, read in bbl file (same for biblatex?)
;; cache this file
;; run latex again if this file changed

(def bibtex-graph
  {:source-dir (fnk [source-file]
                    (fs/normalized (fs/parent source-file)))
   :blxbib-file (fnk [file]
                     (update-ext file "-blx.bib"))
   :start-time
   (fnk [] (System/nanoTime))
   :stop-time
   (fnk [bibtex-run] (System/nanoTime))
   :run-duration
   (fnk [start-time stop-time] (/ (- stop-time start-time ) 1000000.0))
   :copy-blxbib (fnk [blxbib-file source-dir]
                     (when (fs/exists? blxbib-file)
                       (fs/copy blxbib-file (fs/file source-dir
                                                     (fs/base-name blxbib-file)))))
   :bibtex-run (fnk [start-time source-dir file copy-blxbib]
                    (sh/sh "bibtex" (.getPath ^File (fs/normalized file))
                           :dir source-dir
                           ;; add openout_any to environment; otherwise bibtex chokes
                           :env (merge (into {} (System/getenv)) {:openout_any "a"})))
   :bibtex-errors (fnk [bibtex-run]
                       (when (pos? (count (:err bibtex-run)))
                         (helpers/printfln-deco [:red] "[bibtex-error] %s" (:err bibtex-run))))
   :bibtex-warn-prefix (fnk [source-file]
                            (format "[bibtex-warn] source: %s,"
                                    (fs/base-name source-file)))
   :bibtex-warn (fnk [bibtex-run source-file bibtex-warn-prefix]
                     (->> (:out bibtex-run)
                          (re-seq #"Warning--(.*)")
                          (map second)
                          (map (partial str bibtex-warn-prefix))
                          (map (partial helpers/printfln-deco [:yellow]))
                          (dorun)))
   :bibtex-out (fnk [run-duration file-relative]
                    (helpers/printfln-deco
                     [:bold :green]
                     "[bibtex] file: %s, duration: %s"
                     file-relative run-duration))
   :file-relative
   (fnk [file]
        (-> file
            fs/base-name))
   :bbl-file (fnk [file]
                  (update-ext file ".bbl"))
   :new-content (fnk [bbl-file bibtex-errors]
                     (when (fs/exists? bbl-file)
                       (slurp bbl-file)))
   :remove-blxbib (fnk [bibtex-run copy-blxbib]
                       (when copy-blxbib
                         (fs/delete copy-blxbib)))})

#_(fs/copy "sasdas.txt" "asdasdfsdfs.txt")

#_(re-seq #"Warning--(.*)" "klaskdlasd\nWarning--askldsakd\nsakdka sldk a\naskdlksad\nWarning--ksaldkasldkasldk\n")

(defn bibtex [{:keys [file source-file] :as m}]
  (-> (graph-run bibtex-graph m :keep-key? #{:new-content :bbl-file})
      (rename-key :new-content :content)
      (rename-key :bbl-file :file)))

#_(fs/list-dir "/home/david/p/ugcbook/tmp/latex/book/")

#_(bibtex {:file (identity "/home/david/p/ugcbook/tmp/latex/book/assemble.aux")
           :source-file "../../book.tex"})

#_(bibtex {:file (identity "/home/david/p/sdpsize/temp/sdpsize/assemble.aux")
           :source-file "../sdpsize/sdpsize.tex"})

#_(latex-compile/refresh)
