(ns doctex.util
  (:refer-clojure :exclude [assert])
  (:require [clojure
             [spec :as s]
             [walk :as walk]]
            [clojure.java.io :as io]
            [duct.core :as duct]
            [duct.core.env :as env]
            [integrant.core :as ig]
            [taoensso.timbre :as timbre])
  (:import java.io.File
           java.net.URI))

(defn spit-with-parents
  [file content]
  (io/make-parents file)
  (spit file content))

(defn maybe-slurp
  [file]
  (as-> file _
    (io/file _)
    (when (.exists ^File _)
      (slurp _))))

;; use deftype to avoid postwalk expanding val
(deftype Wrap [val])

(def wrap ->Wrap)

(def readers
  {'duct/resource io/resource
   'duct/env      env/env
   'dt/wrap       wrap})

(defn unwrap
  "Unwrap a value, if it is wrapped. Reverses the [[wrap]] function."
  [obj]
  (if (instance? Wrap obj) (.val obj) obj))

(defmethod ig/init-key
  ::unwrap
  [_ v]
  (walk/postwalk unwrap v))

(defn read-config
  ([source]
   (some->> source slurp (ig/read-string {:readers readers})))
  ([source & sources]
   (apply duct/merge-configs (read-config source) (map read-config sources))))

(defn derive-all [derivations]
  (doseq [[tag parent] derivations]
    (derive tag parent)))

(defn relative-path
  "Path of `child` relative to `parent` as string. Returns `nil` if `child` is not a child of `parent`"
  [parent child]
  (let [child-uri ^URI (.toURI (io/file child))
        parent-uri ^URI (.toURI (io/file parent))
        relative-child-uri ^URI (.relativize parent-uri child-uri)]
    (when-not (= child-uri relative-child-uri)
      (.getPath relative-child-uri))))

;;; TODO: move to specs?
(defn assert
  "assert that `data` satisfies spec `s`"
  [s data & [message]]
  (if (s/valid? s data)
    data
    (throw
     (ex-info
      (str (when message
             (format "[%s] " message))
           (s/explain-str s data))
      (->
       (s/explain-data s data)
       (assoc ::data data))))))

;; TODO: assert that var child is unique
(defn find-var-child [k]
  (if (some #{::var} (parents k))
    k
    (some find-var-child (parents k))))

(defn- try-require [sym]
  (try (do (require sym) sym)
       (catch java.io.FileNotFoundException _)))

;; TODO: if a key has ::var as parent, we should automatically update the doc string of the corresponding var

(defmethod ig/init-key
  ::var
  [k v]
  ;; ::var is an ancestor of one of ks
  (let [ks (flatten [k])
        ;; TODO: assert that var child is unique
        k-var (some find-var-child ks)
        ns (namespace k-var)
        _ (try-require (symbol ns))
        sym (symbol ns (name k-var))
        sym-var (find-var sym)]
    (clojure.core/assert sym-var (str "[" k "] var not found:" sym))
    (sym-var k v)))
