(ns doctex.make
  (:require [clojure.java.shell :as sh]
            [clojure.spec :as s]
            [duct.logger :refer [log]]
            [integrant.core :as ig]
            [doctex.core :as core]
            [doctex
             [specs :as specs]
             [util :as util]]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [duct.core :as duct]))

;;; specs

(s/def ::latexcmd
  #{"latexmk"})

(s/def ::output-dir
  ::specs/path-string)

(s/def ::config
  (s/keys
   :req-un [::core/root ::latexcmd ::output-dir ::specs/logger]))

(s/def ::key
  keyword?)

(s/def ::run-config
  (s/and
   ::config
   (s/keys
    :req-un [::key])))

(s/def ::key
  keyword?)

(util/derive-all
 {::config ::util/unwrap
  ::key ::specs/assert})

;;; integrant

(defmethod ig/init-key
  ::content-dir
  [_ {:keys [output-dir root key content-dir]}]
  (.mkdirs (io/file root output-dir (name key) content-dir)))

(defmethod ig/init-key
  ::shell
  [_ {:keys [latexcmd router output-router key logger root] :as config}]
  (log logger :info ::shell [latexcmd key (router key :latex) (output-router key)])
  (sh/sh
   latexcmd
   (str "-output-directory=" (util/relative-path root (output-router key)))
   "-pdf"
   #_"-interaction=nonstopmode"
   "-pdflatex=pdflatex --synctex=1 --src-specials %O %S"
   (util/relative-path root (router key :latex))
   :dir root))

(s/def ::extension
  (s/conformer
   #(get {:log     ".log"
          :pdf     ".pdf"
          :synctex ".synctex.gz"}
         %
         ::s/invalid)))

(defmethod ig/init-key
  ::router
  [_ {:keys [root output-dir logger]}]
  (fn router
    ([k]
     (str
      (io/file
       root
       output-dir
       (name k))))
    ([k ext]
     (log logger :info ::router [k ext])
     (util/assert ::extension ext ::router)
     (str
      (io/file
       root
       output-dir
       (name k)
       (str (name k) (s/conform ::extension ext)))))))

(defmethod ig/init-key
  ::log
  [_ {:keys [router key logger]
      {:keys [err exit out]} :shell}]
  (log logger :info ::log [key (router key) exit])
  (binding [*out* *err*]
    (when (not-empty err)
      (log logger :warn ::log (str/trim err))))
  (log logger :info ::out (str/trim out)))

(defmethod ig/init-key
  ::report
  [_ {:keys [router key logger]}]

  (log logger :report :make (format "%s at %s" key (router key :pdf))))

(s/fdef copy-log
        :args (s/cat :logger ::specs/logger
                     :a ::specs/path-string
                     :b ::specs/path-string))

(defn copy-log [logger a b]
  (let [[file-a file-b] (mapv io/file [a b])]
    (when (.exists file-a)
      (io/make-parents file-b)
      (log logger :info ::copy [a b])
      (io/copy file-a file-b))))

(defmethod ig/init-key
  ::copy
  [_ {:keys [router main-documents shell root logger key output-router]}]
  (copy-log logger
            (output-router key :pdf)
            (router key :pdf))
  (copy-log logger
            (output-router key :synctex)
            (router key :synctex))
  (when (contains? (main-documents) key)
    (log logger :report :copy
         (format "%s to %s" key (io/file root "main.pdf")))
    (copy-log logger
              (output-router key :pdf)
              (str (io/file root "main.pdf")))
    (copy-log logger
              (output-router key :synctex)
              (str (io/file root "main.synctex.gz")))))

(defmethod ig/init-key
  ::run
  [_ {:keys [config]}]
  (fn run [k]
    (->
     config
     (assoc ::key k)
     (duct/prep)
     (ig/init))))

(defmethod ig/init-key
  ::readdocuments
  [k {:keys [readconfig]}]
  (util/assert (s/spec ifn?) readconfig k)

  (fn readdocuments []
    (map first (readconfig))))

(defmethod ig/init-key
  ::main-documents
  [k {:keys [readconfig]}]
  (util/assert (s/spec ifn?) readconfig k)

  (fn readdocuments []
    (->> (readconfig)
         (filter (comp :main second))
         (map first)
         set)))

(defmethod ig/init-key
  ::run-all
  [_ {:keys [readdocuments run]}]
  (doto
      (fn run-all [& _]
        (doseq [k (readdocuments)]
          (run k)))
    (.invoke)))
