(ns doctex.handler
  (:require [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [doctex
             [specs :as specs]
             [util :as util]]
            [duct.logger :refer [log]]
            [integrant.core :as ig]))

(defmethod ig/init-key
  ::combine
  [k handlers]
  (util/assert (s/coll-of ifn?) handlers k)

  (fn combine-handler [files]
    (doseq [handler handlers]
      (handler files))))

(defn starts-with? [files substrings]
  (->>
   (for [file      files
         substring substrings]
     (when (str/starts-with? file substring)
       :match))
   (some #{:match})))

(defmethod ig/init-key
  ::starts-with
  [k {:keys [substrings handler]}]
  (util/assert (s/spec ifn?) handler k)
  (util/assert (s/spec (s/coll-of string?)) substrings k)

  (fn starts-with-handler [files]
    (when (starts-with? files substrings)
      (handler files))))

(defmethod ig/init-key
  ::safe
  [k {:keys [logger error-key handler]}]
  (util/assert ::specs/logger logger k)
  (util/assert (s/spec ifn?) handler k)
  (util/assert keyword? error-key k)

  (fn safe-handler [files]
    (log logger :info ::safe files)
    (try
      (handler files)
      (catch Throwable t
        (log logger :report error-key (util/ex-messages t))
        (log logger :error ::safe {:exception t})))))
