(ns doctex.watch
  (:require [clojure
             [spec :as s]
             [string :as str]]
            [clojure.core.async]
            [clojure.java.io :as io]
            [duct.logger :refer [log]]
            [hawk.core :as hawk]
            [integrant.core :as ig]
            [doctex
             [async :as async]
             [util :as util]]
            [doctex.specs :as specs]
            [duct.core :as duct])
  (:import java.io.File))

;;; specs

(s/def ::watch-path
  (s/and
   string?
   not-empty
   (s/conformer io/file)
   #(.exists ^File %)
   #(.isDirectory ^File %)
   ::specs/canonical-file-conformer))


;;; integrant

(util/derive-all
 {::hawk-ch    ::async/chan-sliding
  ::handler-ch ::async/chan-sliding
  ::throttle   ::async/throttle
  ::chunk      ::async/chunk
  ::handler    ::async/consume})

(defn watch-config [{:keys [path extensions logger handler]}]
  {::hawk           {:path   path
                     :filter (ig/ref ::default-filter)
                     :ch     (ig/ref ::hawk-ch)}
   ::log-watch      {:logger     logger
                     :extensions extensions
                     :path       path}
   ::default-filter {:extensions extensions}
   ::hawk-ch        {}
   ::handler-ch     {}
   ::chunk          {:in  (ig/ref ::hawk-ch)
                     :out (ig/ref ::handler-ch)}
   ::handler        {:ch (ig/ref ::handler-ch)
                     :f  handler}})

(defmethod ig/init-key
  ::hawk
  [_ {:keys [path filter ch]}]

  (util/assert ::watch-path path ::watch-path)

  (hawk/watch!
   [{:paths [(s/conform ::watch-path path)]
     :filter filter
     :handler (fn hawk-handler [_ ev]
                (as-> ev _
                  (:file _)
                  (util/relative-path path _)
                  (clojure.core.async/put! ch _)))}]))

(defmethod ig/halt-key!
  ::hawk
  [_ w]
  (hawk/stop! w))

(util/derive-all
 {::log-watch ::util/var})

(defn log-watch
  "output logging information for watch"
  [k {:keys [path extensions logger]}]
  (log logger :report :watch [path extensions]))

(defmethod ig/init-key
  ::default-filter
  [_ {:keys [extensions]}]
  (fn default-filter
    [_ {:keys [^File file kind]}]
    (and
     (#{:modify :create} kind)
     (.isFile file)
     (some #(str/ends-with? file %) extensions))))

(defmethod ig/init-key
  ::system
  [_ {:keys [extensions path logger handler] :as config}]

  (-> config
      watch-config
      duct/prep
      ig/init))

(defmethod ig/halt-key!
  ::system
  [_ system]
  (ig/halt! system))

