(ns doctex.specs
  (:require [clojure
             [edn :as edn]
             [spec :as s]
             [string :as str]]
            [clojure.java.io :as io]
            [doctex.util :as util]
            duct.logger
            [integrant.core :as ig]
            [yaml.core :as yaml])
  (:import java.io.File))

;;; specs

(s/def ::path-string
  (s/and string?
         not-empty))

(s/def ::join-paths
  (s/and (s/coll-of ::path-string)
         (s/conformer #(str (apply io/file %)))))

(s/def ::append-separator
  (s/and string?
         (s/or :some #(str/ends-with? % File/separator)
               :none (s/and any? (s/conformer #(str % File/separator))))
         (s/conformer second)))

#_(s/describe ::append-separator)

(s/def ::exists-path
  (s/and ::path-string
         #(.exists (io/file %))))

(s/def ::slurp-edn
  (s/and ::exists-path
         (s/conformer #(edn/read-string (slurp %)))))

(s/def ::slurp-yaml
  (s/and ::exists-path
         (s/conformer #(yaml/parse-string (slurp %)))))

(s/def ::fn-slurp-edn
  (s/and ::exists-path
         (s/conformer #(fn [] (edn/read-string (slurp %))))))

(s/def ::canonical-file-conformer
  (s/conformer #(.getCanonicalFile ^File %)))

(defn maybe [s]
  (s/or
   :some s
   :none any?))

(def maybe-conformer
  (s/conformer (fn [[kind v]]
                 (when (#{:some} kind)
                   v))))

(s/def ::maybe-slurp-edn
  (s/and
   ::path-string
   (maybe ::slurp-edn)
   maybe-conformer))

(s/def ::logger
  #(satisfies? duct.logger/Logger %))

(defn get-key [k]
  (s/and
   #(contains? % k)
   (s/conformer #(get % k))))

;;; integrant

(util/derive-all
 {::path-string ::assert
  ::append-separator ::conform
  ::join-paths  ::conform})

(defn find-spec* [k]
  (or (s/get-spec k)
      (some find-spec* (parents k))))

(defn find-spec [k]
  (some find-spec* (flatten [k])))

(defmethod ig/init-key
  ::assert
  [k v]
  (util/assert (find-spec k) v k))

(defmethod ig/init-key
  ::conform
  [k v]
  ;; TODO: better to first conform and throw exception if conforming failed
  ;; current implementation runs the spec always at least twice
  (util/assert (find-spec k) v k)
  (s/conform (find-spec k) v))

(defmethod ig/init-key
  ::conform-fn
  [k v]
  (util/assert (find-spec k) v k)
  (fn []
    (s/conform (find-spec k) v)))
