(ns doctex.logparser
  (:require [cheshire.core :as json]
            [clojure.string :as str]
            [doctex
             [js :as js]
             [util :as util]]
            [duct.logger :refer [log]]
            [integrant.core :as ig]))

(util/derive-all
 {::parse ::util/var
  ::printlog ::util/var})

(defn parse
  [_ {:keys [router key]}]
  (->
   (ig/init
    {::js/engine
     {}

     [::logtext ::js/put]
     {:file    (router key :log)
      :engine  (ig/ref ::js/engine)}

     ::js/get
     {:engine (ig/ref ::js/engine)
      :expression-format
      ["JSON.stringify(%s.parse(%s, {}))" (ig/ref ::latexlogparser) (ig/ref ::logtext)]}

     [::latexlogparser ::js/requirejs-shim]
     {:resource "doctex/vendor/latex-log-parser-sharelatex/dist/latex-log-parser.js"
      :engine   (ig/ref ::js/engine)}})
   ::js/get
   (json/parse-string true)))

(defn format-message [message]
  (->
   message
   (str/replace #" at lines .*" "")
   (str/replace #" at line .*" "")
   (str/replace #" on input line .*" "")
   (str/replace #"on page \d* " "")))

(defn format-level [level]
  (->> level
       (get {"warning"     "WARN: "
             "error"       "ERROR:"
             "typesetting" "TYPE: "})))

(defn keyword-level [level]
  (->> level
       (get {"warning"     :log/warn
             "error"       :log/error
             "typesetting" :log/typeset})))

(defn printlog
  [_ {:keys [parse logger]}]
  (doseq [k [:typesetting :warnings :errors]
          {:keys [level message file line]} (get parse k)]
    (log logger :report (keyword-level level)
         (->>
          [(format-message message) "."
           (when file [" file: " file])
           (when line [": " line])]
          flatten
          (apply str)))))
