(ns deploy.component.webjars
  (:require [com.stuartsierra.component :as component]
            [clojure.java.io :as io]
            [ring.util.codec :as codec]
            [ring.util.request :as req]
            [deploy.serialize :refer [serialize]]
            [clojure.string :as str]
            [clojure.spec :as s]
            [deploy.component.config :as config])
  (:import org.webjars.WebJarAssetLocator))

(def ^:private webjars-pattern
  #"META-INF/resources/webjars/([^/]+)/([^/]+)/(.*)")

(defn- asset-path [prefix resource]
  (when-let [[_ name version path] (re-matches webjars-pattern resource)]
    (str prefix "/" name "/" path)))

(defn- asset-map [^WebJarAssetLocator locator prefix]
  (->> (.listAssets locator "")
       (map (juxt (partial asset-path prefix) identity))
       (remove (comp nil? first))
       (into {})))

(defn- request-path [request]
  (codec/url-decode (req/path-info request)))

(defn- assets
  ([]
   (assets "/assets"))
  ([prefix]
   (asset-map (WebJarAssetLocator.) prefix)))

(defn- copy-assets [prefix destdir xf]
  (doseq [[path resource] (sequence xf (assets prefix))]
    (println "copy" path)
    (io/make-parents (str destdir path))
    (io/copy (-> resource io/resource serialize)
             (io/file (str destdir path)))))



(s/def ::remove-pattern string?)

(s/def ::component
  (s/keys :req-un [::config/common ::remove-pattern]))

(s/fdef exec
        :args (s/cat :component ::component)
        :ret nil?)



(defn exec
  [{{:keys [destdir]} :common
    :keys [remove-pattern]
    :as component}]

  (let [asset-xf (remove (comp (partial re-matches (re-pattern remove-pattern)) first))]

    (println "\n**Copy static assets**")

    (->> asset-xf
         (copy-assets "/assets" destdir)
         (with-out-str)
         (str/split-lines)
         (count)
         (format "(silenced %d lines)")
         (println))))


(defrecord Webjars []
  component/Lifecycle
  (start [component]
    (println "[webjars] start")
    (exec component)
    component)
  (stop [component]
    (println "[webjars] stop")
    component))

(defn webjars [& [m]]
  (map->Webjars m))

#_(component/start-system (load/load-system [(io/resource "deploy.edn")] {})
                          #{:webjars :common})
