(ns component.pandoc
  (:require [clojure.java.shell :as sh]
            [clojure.spec :as s]
            [com.stuartsierra.component :as component]
            [clojure.string :as str]))

;; TODO: check that pandoc is in path
;; TODO: allow for custom pandoc command
;; TODO: add spec for opts and metadata (makes it maybe easier to manipulate)

(s/def ::arg (s/and string? (partial re-matches #"(?s)^--[\-a-z]+(=.*)?")))

#_(s/conform ::arg "--hello-sd-sds=akd123\nsdsd")

(s/def ::args (s/* ::arg))

#_(s/conform ::args ["--hello-sd-sds=akd123\nsdsd" "--hello-sd-sds=akd123\nsdsd" "--hello-sd-sds=akd123\nsdsd"])

(s/def ::metadata-pair (s/cat :key keyword? :values (s/* string?)))

#_(s/conform ::metadata-pair [:hello  "test" "here"])

(s/def ::metadata (s/* (s/spec ::metadata-pair)))

#_(s/conform ::metadata [[:hello  "test" "here"]])

(s/def ::optname (s/and keyword? (comp (partial re-matches #"[a-z\-]+") name)))

#_(s/conform ::optname :sdlk-ksdlk-sd)

#_(s/conform ::optname :sdlk-ksdlk-sd9)

#_(s/conform ::optname :sdlk-ksdlk-sD)

(s/def ::opt (s/cat :name ::optname :value (s/? string?)))

#_(s/conform ::opt [:sdsd-sdsd "skdksd"])

(s/def ::opts (s/* (s/spec ::opt)))

#_(s/conform ::opts [[:sdsd-sdsd "skdksd"]])

(defn- metadata->opts [opts]
  "translate list of options of form [:key \"value\"] to list of metadata options"
  (->> opts
       (mapcat #(map (partial vector (first %)) (rest %)))
       (map #(str (name (first %)) ":" (second %)))
       (map #(identity [:metadata %]))))

(defn- opts->args
  "translate list of options of form [:key \"value\"] to pandoc arguments"
  [opts & {:keys [seps] :or {seps ["--" "="]}}]
  (let [opt->pandoc-arg #(->> %
                              (map name)
                              (interleave seps)
                              (apply str))]
    (map opt->pandoc-arg opts)))

(defn- render*
  "run pandoc on content with provided options and metadata"
  [content opts metadata]
  (let [opts* (concat opts (metadata->opts metadata))
        args (opts->args opts*)
        pandoc-partial (apply partial sh/sh "pandoc" args)
        result (pandoc-partial :in content)]
    (when-not (= 0 (:exit result))
      (throw (ex-info
              "pandoc failed"
              {:function "pandoc"
               :content-length (count content) ;; might be long; just report length
               :err (:err result)
               :opts opts
               :metadata metadata
               :args args})))
    (:out result)))

(defn render
  "render content with opts and metadata, optionally modified by opts-fn and metadata-fn"
  [{:keys [opts metadata] :as component}
   content
   & {:keys [opts-fn metadata-fn]
      :or {opts-fn identity
           metadata-fn identity}}]
  (render* content
           (opts-fn opts)
           (metadata-fn metadata)))

(defrecord Pandoc []
  component/Lifecycle
  (start [this]
    (println "[pandoc] start")
    this)
  (stop [this]
    (println "[pandoc] stop")
    this))

(defn pandoc [& [m]]
  (map->Pandoc m))
