/*
 * Decompiled with CFR 0.152.
 */
package io.dropwizard.checkmate.filters;

import com.codahale.metrics.Gauge;
import com.codahale.metrics.Metric;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.annotation.Metered;
import com.google.common.base.Strings;
import com.google.common.cache.LoadingCache;
import com.google.common.io.ByteStreams;
import io.dropwizard.checkmate.CheckmateBundle;
import io.dropwizard.checkmate.ChecksumAlgorithm;
import io.dropwizard.checkmate.core.CheckmateError;
import io.dropwizard.checkmate.core.ChecksumInfo;
import io.dropwizard.checkmate.model.CheckmateBundleConfiguration;
import io.dropwizard.checkmate.utils.AESUtil;
import io.dropwizard.checkmate.utils.ChecksumUtil;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import javax.annotation.Priority;
import javax.inject.Singleton;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.core.Response;
import javax.ws.rs.ext.Provider;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Provider
@Singleton
@Priority(value=1000)
public class CheckmateRequestFilter
implements ContainerRequestFilter {
    private static final Logger log = LoggerFactory.getLogger(CheckmateRequestFilter.class);
    private static final int INDEX_OF_APIS = 5;
    private CheckmateBundleConfiguration configuration;
    private final LoadingCache<String, String> appSignatureCache;
    private final Map<String, Long> metricNameVsTimeTakenMap = new ConcurrentHashMap<String, Long>();

    public CheckmateRequestFilter(CheckmateBundleConfiguration configuration, LoadingCache<String, String> appSignatureCache, MetricRegistry metrics) {
        this.configuration = configuration;
        this.appSignatureCache = appSignatureCache;
        configuration.getPathVsMetricMap().forEach((path, metric) -> metrics.remove(String.format("checkmate.latency.%s", metric)));
        configuration.getPathVsMetricMap().forEach((path, metric) -> {
            try {
                metrics.register(String.format("checkmate.latency.%s", metric), (Metric)new Gauge<Long>(){

                    public Long getValue() {
                        return (Long)CheckmateRequestFilter.this.metricNameVsTimeTakenMap.get(metric);
                    }
                });
            }
            catch (Exception e) {
                log.error("Error occurred while registering the metric : " + metric);
            }
        });
    }

    @Metered(name="checkmate")
    public void filter(ContainerRequestContext requestContext) throws IOException {
        long start = System.currentTimeMillis();
        String path = requestContext.getUriInfo().getPath();
        boolean isWhiteListed = this.isWhitelisted(path);
        String v1Token = requestContext.getHeaderString("X-REQUEST-TOKEN");
        String v2Token = requestContext.getHeaderString("X-REQUEST-CHECKSUM");
        String sdkChecksum = requestContext.getHeaderString("X-REQUEST-SDK-CHECKSUM");
        if (!Strings.isNullOrEmpty((String)sdkChecksum)) {
            this.validateSDKChecksum(requestContext, path, sdkChecksum, this.configuration);
            this.registerMetric(requestContext, System.currentTimeMillis() - start);
            return;
        }
        if (!Strings.isNullOrEmpty((String)v2Token)) {
            this.validateV2Token(requestContext, path, v2Token, this.configuration);
            this.registerMetric(requestContext, System.currentTimeMillis() - start);
            return;
        }
        if (!Strings.isNullOrEmpty((String)v1Token)) {
            this.validateV1Token(requestContext, path, v1Token, this.configuration);
            this.registerMetric(requestContext, System.currentTimeMillis() - start);
            return;
        }
        if (isWhiteListed) {
            log.info("Let through whitelisted API: {}", (Object)path);
            this.registerMetric(requestContext, System.currentTimeMillis() - start);
            return;
        }
        log.info("API is not whitelisted and no checksum header present for {}", (Object)path);
        requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)CheckmateError.builder().errorCode("CK001").message("Bad request").build()).build());
        this.registerMetric(requestContext, System.currentTimeMillis() - start);
    }

    private void registerMetric(ContainerRequestContext requestContext, long timeTaken) {
        String path = requestContext.getUriInfo().getPath().substring(5);
        String[] metricName = new String[1];
        this.configuration.getPathVsMetricMap().forEach((k, v) -> {
            if (path.matches((String)k)) {
                metricName[0] = v;
            }
        });
        if (StringUtils.isEmpty((CharSequence)metricName[0])) {
            return;
        }
        this.metricNameVsTimeTakenMap.put(metricName[0], timeTaken);
    }

    private void validateSDKChecksum(ContainerRequestContext requestContext, String path, String token, CheckmateBundleConfiguration configuration) {
        block9: {
            ChecksumInfo checksumInfo;
            try {
                checksumInfo = ChecksumUtil.decodeHeaderSDK(token);
                checksumInfo.setChecksumAlgorithm(ChecksumAlgorithm.ALGO_SDK_V1);
            }
            catch (Exception e) {
                log.error("Bad checksum header: " + path, (Throwable)e);
                if (configuration.isRequestValidation()) {
                    requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)CheckmateError.builder().errorCode("CK001").message("Bad request").build()).build());
                }
                return;
            }
            String appId = requestContext.getHeaderString("X-APP-ID");
            if (Strings.isNullOrEmpty((String)appId)) {
                requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)CheckmateError.builder().errorCode("CK006").message("Bad request").build()).build());
                return;
            }
            String appKey = null;
            try {
                appKey = (String)this.appSignatureCache.get((Object)appId);
            }
            catch (Throwable t) {
                log.error("Error loading key for " + appId, t);
                requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)CheckmateError.builder().errorCode("CK007").message("Bad request").build()).build());
                return;
            }
            if (Strings.isNullOrEmpty((String)appKey)) {
                requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)CheckmateError.builder().errorCode("CK007").message("Bad request").build()).build());
                return;
            }
            String encryptionKey = ChecksumUtil.generateKey(appKey, checksumInfo.getSalt());
            try {
                log.debug("Encryption key: {} Payload: {}", (Object)encryptionKey, (Object)checksumInfo.getChecksumData());
                String checksumData = AESUtil.decrypt(checksumInfo.getChecksumData(), encryptionKey);
                checksumInfo.setEncryptionKey(encryptionKey);
                CheckmateRequestFilter.validateBody(requestContext, path, configuration, checksumInfo, checksumData);
            }
            catch (Exception e) {
                this.logException(requestContext, configuration, checksumInfo, path, appId, e);
                if (!configuration.isRequestValidation()) break block9;
                requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)CheckmateError.builder().errorCode("CK005").message("Bad request").build()).build());
            }
        }
    }

    private void validateV2Token(ContainerRequestContext requestContext, String path, String token, CheckmateBundleConfiguration configuration) {
        block9: {
            ChecksumInfo checksumInfo;
            try {
                checksumInfo = ChecksumUtil.decodeHeader(token);
                checksumInfo.setChecksumAlgorithm(ChecksumAlgorithm.ALGO_V2);
            }
            catch (Exception e) {
                log.error("Bad checksum header: " + path, (Throwable)e);
                if (configuration.isRequestValidation()) {
                    requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)CheckmateError.builder().errorCode("CK001").message("Bad request").build()).build());
                }
                return;
            }
            String appId = requestContext.getHeaderString("X-APP-ID");
            if (Strings.isNullOrEmpty((String)appId)) {
                requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)CheckmateError.builder().errorCode("CK006").message("Bad request").build()).build());
                return;
            }
            String appKey = null;
            try {
                appKey = (String)this.appSignatureCache.get((Object)appId);
            }
            catch (Throwable t) {
                log.error("Error loading key for " + appId, t);
                requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)CheckmateError.builder().errorCode("CK007").message("Bad request").build()).build());
                return;
            }
            if (Strings.isNullOrEmpty((String)appKey)) {
                requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)CheckmateError.builder().errorCode("CK007").message("Bad request").build()).build());
                return;
            }
            String encryptionKey = String.format("%s%s", checksumInfo.getSalt(), appKey);
            try {
                log.debug("Key: {} Payload: {}", (Object)encryptionKey, (Object)checksumInfo.getChecksumData());
                String checksumData = AESUtil.decrypt(checksumInfo.getChecksumData(), encryptionKey);
                checksumInfo.setEncryptionKey(encryptionKey);
                CheckmateRequestFilter.validateBody(requestContext, path, configuration, checksumInfo, checksumData);
            }
            catch (Exception e) {
                this.logException(requestContext, configuration, checksumInfo, path, appId, e);
                if (!configuration.isRequestValidation()) break block9;
                requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)CheckmateError.builder().errorCode("CK005").message("Bad request").build()).build());
            }
        }
    }

    private void logException(ContainerRequestContext requestContext, CheckmateBundleConfiguration configuration, ChecksumInfo checksumInfo, String path, String appId, Exception e) {
        StringBuilder sb = new StringBuilder();
        for (String header : configuration.getHeadersList()) {
            String headerValue = ", " + header + " : " + requestContext.getHeaderString(header);
            sb.append(headerValue);
        }
        log.error("ErrorDecrypting token: appId : " + appId + ", path : " + path + ", headers : " + sb.toString() + checksumInfo, (Throwable)e);
    }

    private void validateV1Token(ContainerRequestContext requestContext, String path, String token, CheckmateBundleConfiguration configuration) {
        block5: {
            ChecksumInfo checksumInfo;
            try {
                checksumInfo = ChecksumUtil.decodeHeader(token);
                checksumInfo.setChecksumAlgorithm(ChecksumAlgorithm.ALGO_V1);
            }
            catch (Exception e) {
                log.error("Bad checksum header: " + path, (Throwable)e);
                if (configuration.isRequestValidation()) {
                    requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)CheckmateError.builder().errorCode("CK001").message("Bad request").build()).build());
                }
                return;
            }
            try {
                checksumInfo.setEncryptionKey(checksumInfo.getSalt());
                String checksumData = AESUtil.decrypt(checksumInfo.getChecksumData(), checksumInfo.getEncryptionKey());
                CheckmateRequestFilter.validateBody(requestContext, path, configuration, checksumInfo, checksumData);
            }
            catch (Exception e) {
                this.logException(requestContext, configuration, checksumInfo, path, "", e);
                if (!configuration.isRequestValidation()) break block5;
                requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)CheckmateError.builder().errorCode("CK005").message("Bad request").build()).build());
            }
        }
    }

    private static void validateBody(ContainerRequestContext requestContext, String path, CheckmateBundleConfiguration configuration, ChecksumInfo checksumInfo, String checksumData) throws IOException {
        String[] checksumTokens = checksumData.split("###");
        if (checksumTokens.length != 2) {
            log.error("Invalid checksum: {} {}", (Object)path, (Object)checksumData);
            if (configuration.isRequestValidation()) {
                requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)CheckmateError.builder().errorCode("CK002").message("Bad request").build()).build());
            }
            return;
        }
        long requestTime = Long.parseLong(checksumTokens[0]);
        if (configuration.isExpiryCheck() && ChecksumUtil.isExpired(requestTime, configuration.getClockSkew())) {
            log.error("Request token expired: {} {}", (Object)path, (Object)checksumData);
            if (configuration.isRequestValidation()) {
                requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)CheckmateError.builder().errorCode("CK003").message("Bad request").build()).build());
            }
            return;
        }
        byte[] body = ByteStreams.toByteArray((InputStream)requestContext.getEntityStream());
        requestContext.setEntityStream((InputStream)new ByteArrayInputStream(body));
        String checksum = checksumTokens[1];
        String computedChecksum = ChecksumUtil.checksum(requestContext.getUriInfo().getRequestUri().getPath(), body);
        if (!checksum.equals(computedChecksum)) {
            log.error("Request checksum invalid: {} [Checksum: {}, Computed: {}]", new Object[]{path, checksum, computedChecksum});
            if (configuration.isRequestValidation()) {
                requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)CheckmateError.builder().errorCode("CK004").message("Bad request").build()).build());
            }
            return;
        }
        log.debug("Checksum validation successful: " + checksumInfo);
        requestContext.setProperty("io.dropwizard.checkmate.filters.checksumInfo", (Object)checksumInfo);
        requestContext.setProperty("io.dropwizard.checkmate.filters.checksum", (Object)checksum);
    }

    private boolean isWhitelisted(String path) {
        return CheckmateBundle.getWhiteList().stream().anyMatch(path::matches);
    }

    public static CheckmateRequestFilterBuilder builder() {
        return new CheckmateRequestFilterBuilder();
    }

    public static class CheckmateRequestFilterBuilder {
        private CheckmateBundleConfiguration configuration;
        private LoadingCache<String, String> appSignatureCache;
        private MetricRegistry metrics;

        CheckmateRequestFilterBuilder() {
        }

        public CheckmateRequestFilterBuilder configuration(CheckmateBundleConfiguration configuration) {
            this.configuration = configuration;
            return this;
        }

        public CheckmateRequestFilterBuilder appSignatureCache(LoadingCache<String, String> appSignatureCache) {
            this.appSignatureCache = appSignatureCache;
            return this;
        }

        public CheckmateRequestFilterBuilder metrics(MetricRegistry metrics) {
            this.metrics = metrics;
            return this;
        }

        public CheckmateRequestFilter build() {
            return new CheckmateRequestFilter(this.configuration, this.appSignatureCache, this.metrics);
        }

        public String toString() {
            return "CheckmateRequestFilter.CheckmateRequestFilterBuilder(configuration=" + this.configuration + ", appSignatureCache=" + this.appSignatureCache + ", metrics=" + this.metrics + ")";
        }
    }
}

