; Licensed to the Apache Software Foundation (ASF) under one
; or more contributor license agreements. See the NOTICE file
; distributed with this work for additional information
; regarding copyright ownership. The ASF licenses this file
; to you under the Apache License, Version 2.0 (the
; "License"); you may not use this file except in compliance
; with the License. You may obtain a copy of the License at
;
; http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.

(ns org.domaindrivenarchitecture.pallet.servertest.test
  (:require
    [clojure.tools.logging :as logging]
    [schema.core :as s]
    [pallet.crate :as crate]
    [pallet.actions :as actions]
    [pallet.stevedore :refer :all]
    [org.domaindrivenarchitecture.pallet.servertest.fact :refer :all]))

(s/defn format-test-result
  "Creates the session output."
  [fact-key :- s/Keyword
   test-result :- s/Bool
   writer-output :- s/Str]
  {:context "server-test"
   :action-symbol (str "test " fact-key)
   :out writer-output
   :exit (if test-result 0 -1)
   :summary (if test-result "SUCCESFUL" (str "TEST FAILD: " writer-output))})

(s/defn server-test
  "If fact is not available, then execute fact collection once & test results. 
   The given test-fn should:
     * consume the following parameters:
         [fact-data,
          test-writer]
     * evaluate to the test-result, which means:
         true for sucessfull passed tests & 
         false for failed tests.
     * additional information can be written to the provided writer (e.g. using println).
       If desired, this will show up in the test result"
  {:pallet/plan-fn true}
  [fact-key :- s/Keyword 
   test-fn]
  (actions/as-action
    (let [writer (new java.io.StringWriter)
          session-data (crate/get-settings :dda-servertest-fact)
          fact-session-data (fact-key session-data)
          fact-data (if (some? (:out fact-session-data)) fact-session-data
                      (merge 
                        fact-session-data
                        (format-fact-data 
                          (actions/exec-script ~(:script fact-session-data)) 
                          (:transform-fn fact-session-data))))]
      (crate/update-settings :dda-servertest-fact {fact-key fact-data})
      (actions/as-action
        (logging/info "server-test: using fact" fact-key "with" (:out fact-data)))
      (let [test-result (format-test-result fact-key (apply test-fn (:out fact-data) writer))]
        (crate/assoc-settings :dda-servertest-test fact-key test-result)
        (actions/as-action
          (logging/info "server-test: " fact-key "was" (:summary test-result)))
        test-result)
      ))
  )
