; Licensed to the Apache Software Foundation (ASF) under one
; or more contributor license agreements. See the NOTICE file
; distributed with this work for additional information
; regarding copyright ownership. The ASF licenses this file
; to you under the Apache License, Version 2.0 (the
; "License"); you may not use this file except in compliance
; with the License. You may obtain a copy of the License at
;
; http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.

(ns org.domaindrivenarchitecture.pallet.crate.d2rq.app
   (:require
     [pallet.actions :as actions]
     [pallet.stevedore :as stevedore]
     [clojure.string :as string]
    ))

(defn install-d2rq-war-file
  [tomcat-dir linked-data-war-url 
   & {:keys [webapp-name]
      :or {webapp-name "d2rq"}}]
  (actions/remote-file
    (str tomcat-dir "webapps/" webapp-name ".war") 
    :action :create
    :url linked-data-war-url
    :owner "tomcat7" ;assumes existing tomcat7 user
    :group "tomcat7"
    :mode "755"
    )
  )

(defn- mapping-header-lines
  [rdf-label
   fqdn
   d2rq-port
   webapp-name
   db-name
   db-user-name
   db-user-passwd]
  ["@prefix map: <#> ."
   "@prefix db: <> ."
   "@prefix vocab: <vocab/> ."
   "@prefix rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#> ."
   "@prefix rdfs: <http://www.w3.org/2000/01/rdf-schema#> ."
   "@prefix xsd: <http://www.w3.org/2001/XMLSchema#> ."
   "@prefix d2rq: <http://www.wiwiss.fu-berlin.de/suhl/bizer/D2RQ/0.1#> ."
   "@prefix jdbc: <http://d2rq.org/terms/jdbc/> ."
   "@prefix d2r: <http://sites.wiwiss.fu-berlin.de/suhl/bizer/d2r-server/config.rdf#> ."
   ""
   "<> a d2r:Server;"
   (str " rdfs:label \"" rdf-label "\";")
   (str " d2r:baseURI <https://" fqdn ":" d2rq-port "/" webapp-name "/" ">;")
   (str " d2r:port " d2rq-port ";")
   " d2r:vocabularyIncludeInstances true;"
   " d2r:limitPerClassMap 2500;"
   " d2r:limitPerPropertyBridge 2500;"
   " d2r:sparqlTimeout -1;"
   "."
   ""
   "map:database a d2rq:Database;"
   "	d2rq:resultSizeLimit 5000;"
   "	d2rq:jdbcDriver \"com.mysql.jdbc.Driver\";"
   (str "	d2rq:jdbcDSN \"jdbc:mysql://127.0.0.1:3306/" db-name "\";")
   (str "	d2rq:username \"" db-user-name "\";")
   (str "	d2rq:password \"" db-user-passwd "\";")
   "	jdbc:autoReconnect \"true\";"
   "	jdbc:zeroDateTimeBehavior \"convertToNull\";"
   "."]
  )


(defn deploy-ontology-pa
  [tomcat-dir 
   d2rq-dir 
   linked-data-download-url
   header-file-name
   rdf-label
   fqdn
   d2rq-port
   db-name
   db-user-name
   db-user-passwd
   & {:keys [webapp-name]
        :or {webapp-name "d2rq"}}]
  (let [d2rq-webapp (str tomcat-dir "/webapps/" webapp-name)
        ttl-file (str d2rq-webapp "/WEB-INF/config-example.ttl")]

   ;unzip war file into tomcat/webapps folder 
  (actions/remote-directory
    d2rq-webapp
    :remote-file (str d2rq-webapp ".war")
    :unpack :unzip
    :recursive true 
    :mode "755"
    :owner "tomcat7"
    :group "tomcat7"
    )
  
  ;get ttls to be published
  (actions/remote-directory
    (str d2rq-dir "/ttl-sources")
    :unpack :unzip
    :url linked-data-download-url
    :owner "tomcat7" ;assumes existing tomcat7 user
    :group "tomcat7"
    :mode "755"
    )
  
  ;overwrite header
  (actions/remote-file
    (str d2rq-dir "/ttl-sources/" header-file-name) 
    :owner "tomcat7"
    :group "tomcat7"
    :overwrite-changes true
    :literal true
    :content 
    (string/join
      \newline
      (mapping-header-lines
        rdf-label
        fqdn
        d2rq-port
        webapp-name
        db-name
        db-user-name
        db-user-passwd)))
  
  ;delete existing config-example.ttl
  (actions/file
    ttl-file
    :action :delete
    )
  
  ;create empty config-example.ttl
  (actions/file
    ttl-file
    :action :create
    :mode "644"
    :owner "tomcat7"
    :group "tomcat7")
  
  ;concat ttl files into config-example.ttl
  (actions/exec
    {:language :bash}
    (stevedore/script
      (str "for f in " (d2rq-dir) "/ttl-sources/*.ttl ; do (cat \"${f}\"; echo) >> " (ttl-file) " ; done")
      ))
  
  )   
  
) 