; Licensed to the Apache Software Foundation (ASF) under one
; or more contributor license agreements. See the NOTICE file
; distributed with this work for additional information
; regarding copyright ownership. The ASF licenses this file
; to you under the Apache License, Version 2.0 (the
; "License"); you may not use this file except in compliance
; with the License. You may obtain a copy of the License at
;
; http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.
(ns org.domaindrivenarchitecture.validators-test
  (:require
    [clojure.test :refer :all]
    [org.domaindrivenarchitecture.validatable :as validateable]
    [org.domaindrivenarchitecture.valid-mock :as mock]
    [org.domaindrivenarchitecture.validators :as sut]
    ))

(deftest get-value-from-path
  (testing
    "test the path thru a map structure"
    (is (= 1
           (sut/get-value-from-path {:x {:y 1}} :x :y))
        ))
  (testing
    "non existing path"
    (is (thrown? AssertionError
           (sut/get-value-from-path {:q "1"} :x :y))
        ))
  (testing
    "graph should provide map protocol"
    (is (thrown? AssertionError
           (sut/get-value-from-path "" :x :y))
        ))
  )

(deftest validate-not-empty
  (testing
    "test the path thru a map structure"
    (is (= []
           (sut/validate-not-empty {:x {:y 1}} :x :y))
        ))
  (testing
    "test not existing path"
    (is (= ["vallidation-not-empty: (:x :n)"]
           (sut/validate-not-empty {:x {:y 1}} :x :n))
        )
    (is (= [(str "vallidation-not-empty: (:x :n)\n"
            "java.lang.AssertionError: Assert failed: (or (empty? path) (map? graph))")]
             (sut/validate-not-empty {:x "1"} :x :n))
          ))
  )

(deftest validateable
  (testing 
    "does the mock work?"
    (is (satisfies? validateable/Validatable (mock/new-valid-mock true)))
    (is (validateable/valid? (mock/new-valid-mock true)))
    (is (not (validateable/valid? (mock/new-valid-mock false))))
    )
  )

(deftest validate-map
  (testing
    "test validation of a map"
    (is (= []
           (sut/validate-map {:x (mock/new-valid-mock true)
                              :y (mock/new-valid-mock true)}))
        )
    (is (= ["mock invalid"]
           (sut/validate-map {:x (mock/new-valid-mock false)
                              :y (mock/new-valid-mock true)}))
        )
    (is (= ["mock invalid"
            "mock invalid"]
       (sut/validate-map {:x (mock/new-valid-mock false)
                          :y (mock/new-valid-mock true)
                          :z (mock/new-valid-mock false)}))
    )
     (is (= ["not validateable :z, not validatable"]
           (sut/validate-map {:x (mock/new-valid-mock true)
                              :y (mock/new-valid-mock true)
                              :z "not validatable"}))
        )
    )
  )