; Licensed to the Apache Software Foundation (ASF) under one
; or more contributor license agreements. See the NOTICE file
; distributed with this work for additional information
; regarding copyright ownership. The ASF licenses this file
; to you under the Apache License, Version 2.0 (the
; "License"); you may not use this file except in compliance
; with the License. You may obtain a copy of the License at
;
; http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.

(ns org.domaindrivenarchitecture.pallet.crate.config.config-provider
  (:require
    [pallet.api :as api]
    [pallet.crate :as crate]
    [pallet.actions :as actions]
    [pallet.stevedore :as stevedore]
    [org.domaindrivenarchitecture.validatable :as validatable]
    [org.domaindrivenarchitecture.validators :as validators]
    ))

(defn version-detection
  [facility]
  (stevedore/script
    (println "{")
    (println " :facility" (str "'\"'" ~facility "'\"'"))
    (println " :version-installed" 
             (str "'\"'" @("cat" ~(str "/var/lib/pallet/versions-installed/" facility)) "'\"'"))
    (println "}")))

(defprotocol ConfigProvider
  (remote-version [facility node-id])
  (global-config [config-provider])
  (nodespecific-config [config-provider node-id])
  )

(defrecord ConfigProviderRecord
  [global node-specific]
  validatable/Validatable
  (validatable/valid? [this] (validatable/valid? this nil))
  (validatable/valid? [this context] 
    (empty? (validatable/validate this context)))
  (validatable/validate [this] (validatable/validate this nil))
  (validatable/validate [this context]
    (into []
          (concat
            (validators/validate-not-empty this :global)
            (validators/validate-not-empty this :node-specific)
            (validators/validate-map (:global this))
            (validators/validate-map (:node-specific this))
            ))
    )          
  ConfigProvider
  (remote-version [facility node-id]
    (actions/exec-script (version-detection facility)))
  (global-config [this]
    (-> this :global))
  (nodespecific-config [this node-id] 
    (-> this :node-specific node-id))
  )

(defn detect-remote-version [facility node-id]
  (let [res (actions/exec-script* (version-detection facility))]
    (actions/with-action-values
      [res]
      (-> res :out)    
      )
    ))

(defn new-config-provider 
  [& {:keys [global
             node-specific]
      :as parameters}]
  {:pre [(= 2 (count parameters))]
   :post [(validatable/valid? %)]
   }
  (ConfigProviderRecord. global node-specific))