; Licensed to the Apache Software Foundation (ASF) under one
; or more contributor license agreements. See the NOTICE file
; distributed with this work for additional information
; regarding copyright ownership. The ASF licenses this file
; to you under the Apache License, Version 2.0 (the
; "License"); you may not use this file except in compliance
; with the License. You may obtain a copy of the License at
;
; http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.

(ns org.domaindrivenarchitecture.pallet.crate.hosteurope
  (:require
    [pallet.actions :as actions]
    [pallet.api :as api]    
    [pallet.crate :as crate]
    [pallet.node :as node]
    [pallet.stevedore :as stevedore]
    [org.domaindrivenarchitecture.pallet.crate.config :as config]
    [org.domaindrivenarchitecture.cm.util :as util]
))

(def facility :dda-hosteurope)

;; remove mysql in order to be able to set root password
;; sudo apt-get --purge remove mysql-server mysql-common mysql-client

(defn configure-limits
  "reflecting values from https://faq.hosteurope.de/index.php?cpid=13281"
  [hosterope-server-type]
  (let [max-files 
        (case hosterope-server-type
          :starter 17600
          :advanced 17600
          :expert 24672
          :unlimited 38496
          )
        file-limits 
        [(str "root	soft	nofile " max-files)
         (str "root	hard	nofile  " max-files)
         (str "*	soft	nofile  " max-files)
         (str "*	hard	nofile  " max-files)]]    
    (actions/remote-file "/etc/security/limits.d/hosteurope.conf"
                 :owner "root" 
                 :group "root" 
                 :mode "0644" 
                 :literal true
                 :content (util/create-file-content file-limits)
    )
  ))

(defn prepare-mysql
  []
  (actions/file
    "/etc/init/mysql.override"
    :action :delete
    )
  (actions/exec
        {:language :bash}
        (stevedore/script
          ("apt-get -y remove mysql-* --purge")))
  )

(defn configure-hostname
  ""
  [ip hostname domainname]
  (let [fqdn (str hostname "." domainname)]
    (actions/remote-file "/etc/hosts"
                 :owner "root" :group "root" :mode "0644" :literal true
                 :content (str "127.0.0.1 localhost.localdomain localhost\n"
                               ip " " fqdn " " hostname))
    (actions/remote-file "/etc/hostname"
                 :owner "root" :group "root" :mode "0644" :literal true
                 :content fqdn)
    (actions/plan-when-not 
      (stevedore/script (file-exists? "/etc/init.d/hostname_vps"))
      (actions/remote-file "/etc/init.d/hostname_vps"
                           :literal true
                           :owner "root" 
                           :group "root" 
                           :mode "0744"
                           :content 
                           (stevedore/script
                             "#!/bin/bash"
                             "### BEGIN INIT INFO"
                             "# Provides: hostname_vps"
                             "# Required-Start:"
                             "# Required-Stop:"
                             "# Default-Start: 2 3 4 5"
                             "# Default-Stop: 0 6"
                             "# Short-Description: set hostname."
                             "# Description: Host Europe virtual server hostname setting script."
                             "### END INIT INFO"
                             "#"
                             "# chkconfig: 2345 7 93"
                             "# description: Host Europe virtual server hostname setting script"
                             "#"
                             "# /etc/init.d/hostname_vps"
                             "#"
                             (str "HOSTNAME=" ~fqdn)
                             ("[ -f /etc/hostname ] && echo $HOSTNAME > /etc/hostname")
                             ("[ -f /etc/HOSTNAME ] && echo $HOSTNAME > /etc/HOSTNAME")
                             ("[ -f /etc/sysconfig/network ] && sed -i.hebackup s/HOSTNAME=\".*\"/HOSTNAME=\\\"$HOSTNAME\\\"/ /etc/sysconfig/network")
                             ("/bin/hostname $HOSTNAME")
                             ("[ -f /etc/my.resolv.conf ] && cp /etc/my.resolv.conf /etc/resolv.conf")
                             ("[ -f /etc/my.hosts ] && cp /etc/my.hosts /etc/hosts")))
      (actions/exec
        {:language :bash}
        (stevedore/script
          ("/usr/sbin/update-rc.d hostname_vps defaults 09"))
        ))
    )
  )

(def with-hosteurope
  ^:dynamic
  (api/server-spec
    :phases 
    {:init
     (api/plan-fn
       (let [ip (node/primary-ip (crate/target-node))
             hostname-from-config
             (-> (config/get-nodespecific-config) :host-name)
             domainname-from-config
             (-> (config/get-nodespecific-config) :domain-name)
             server-type
             (:server-type 
               (config/get-nodespecific-additional-config facility))]
         (configure-hostname ip hostname-from-config domainname-from-config)
         (configure-limits server-type)
         (prepare-mysql)
       ))
     })
  )