(ns dda.build.terragrunt.domain
  (:require [clojure.spec.alpha :as s]
            [orchestra.core :refer [defn-spec]]
            [dda.build.devops.domain :as d]
            [dda.c4k-common.predicate :as pred]))

(s/def ::sensitive boolean?)

(s/def ::type vector?)

(s/def ::fqdn pred/fqdn-string?)
(s/def ::ipv4 pred/ipv4-string?)
(s/def ::ipv6 pred/ipv6-string?)
(s/def ::value
  (s/keys :req-un [::fqdn ::ipv4 ::ipv6]))

(s/def ::out
  (s/keys :req-un [::sensitive ::type ::value]))

(s/def ::tf-out
  (s/keys :req-un [::out]))

(s/def ::tg-output-filenname string?)

(s/def ::config
  (s/merge ::d/devops
           (s/keys :req-un [::tg-output-filenname]
                   :opt-un [])))

(defn-spec clean-build-dir-command seq?
  [config ::config]
  ["rm"  "-rf" (d/build-path (dissoc config :module))])

(defn-spec copy-terragrunt-command seq?
  [config ::config]
  (let [{:keys [module]} config
        config-wo-module (dissoc config :module)]
    [["bash" "-c" (str "cp *.hcl " (d/build-path config-wo-module))]
     ["cp" "-r" module (d/build-path config-wo-module)]]))

(defn-spec terragrunt-plan-command seq?
  [config ::config]
  [[{:dir (d/build-path config)} "terragrunt" "init"]
   [{:dir (d/build-path config)} "terragrunt" "plan"]])

(defn-spec terragrunt-apply-command seq?
  [config ::config]
  [[{:dir (d/build-path config)} "terragrunt" "init"]
   [{:dir (d/build-path config)} "terragrunt" "apply" "-auto-approve"]])

(defn-spec terragrunt-destroy-command seq?
  [config ::config]
  [[{:dir (d/build-path config)} "terragrunt" "init"]
   [{:dir (d/build-path config)} "terragrunt" "destroy"]])

(defn-spec terragrunt-output-command seq?
  [config ::config]
  (let [{:keys [tg-output-filenname]} config]
    [[{:dir (d/build-path config)} "bash" "-c" (str "terragrunt output -json > " tg-output-filenname)]]))

(defn-spec output-path string?
  [config ::config]
  (let [{:keys [tg-output-filenname]} config]
    (str (d/build-path config) "/" tg-output-filenname)))
