; Licensed to the Apache Software Foundation (ASF) under one
; or more contributor license agreements. See the NOTICE file
; distributed with this work for additional information
; regarding copyright ownership. The ASF licenses this file
; to you under the Apache License, Version 2.0 (the
; "License"); you may not use this file except in compliance
; with the License. You may obtain a copy of the License at
;
; http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.

(ns org.domaindrivenarchitecture.pallet.crate.versions
  (:require
      [pallet.actions :as actions]
      [pallet.stevedore :as stevedore]
      [pallet.script :as script]
      [pallet.crate :as crate]
      [pallet.api :as api]
      [pallet.node-value :as nv]
      [pallet.core.session :refer [session session! *session*]]
      
      [clojure.tools.logging :as logging]
      [clojure.string :refer [join]]
      ))


(defn- ver_str2int
  "Converts string to integer, ignores all characters but 0-9. 
   Returns 0 if string is empty."
  [str]
  (if (= 0 (count str))
    0
    (Integer. (re-find  #"\d+" str ))))

(defn ver_fromstr
  "Converts formated version string (1.2.3.4) to version vector [1 2 3 4]"
  [str]
  (into [] (map ver_str2int (clojure.string/split str #"\."))))

(defn ver_str [version]
  "Converts verstion vector to point seperated string.
   Example (= (ver_str [2 1 4]) \"2.1.4\") "
  (join "." version))

(defn- ver_fill [ver length]
  "Fills up a version vector with trailing zeros.
   Example (= (verfill [2 1] 4) [2 1 4 4]) "
  (if (> length (count ver))
    (concat ver (repeat (- length (count ver)) 0))
    ver
  ))

(defn- ver_comp [v1 v2]
  "Compares two version vectors and return the difference of the first postion they differ.
   Returns nil if they are the same version."
  (let [len (max (count v1) (count v2))]
   (first (drop-while #(= % 0) (mapv - (ver_fill v1 len) (ver_fill v2 len))))
  ))

(defn- ver_less [v1 v2]
  "Returns v1 < v2"
  (let [comp (ver_comp v1 v2)]
    (if (nil? comp) false (< comp 0))))
(defn- ver_lesseq [v1 v2]
  "Returns v1 <= v2"
  (let [comp (ver_comp v1 v2)]
    (if (nil? comp) true (< comp 0))))
(defn- ver_eq [v1 v2]
  "Returns v1 == v2"
  (let [comp (ver_comp v1 v2)]
    (nil? comp)))
    

(defprotocol VersionSelector
  "Simple protocol for a selector that compares versions. This is used in base crate to
   identify selector objects."
  (compare-versions [this vremote vpallet]))
(defrecord ver-lesseq-selector [verfrom]
  VersionSelector
  (compare-versions [this vremote vpallet] 
    (and (not (ver_eq vremote vpallet))
         (not (nil? vremote)) 
         (ver_lesseq vremote (:verfrom this)))))
(defrecord ver-notinstalled-selector []
  VersionSelector
  (compare-versions [this vremote vpallet] 
    (nil? vremote)))
(defrecord ver-always []
  VersionSelector
  (compare-versions [this vremote vpallet] 
    true))

(defn ver-lesseq? [verfrom]
  "Creates version selector for less or equal a specified version."
  (->ver-lesseq-selector verfrom))

(defn ver-notinstalled? []
  "Creates version selector for remote systems that have no version of the app installed."
  (->ver-notinstalled-selector))

(defn ver-always? []
  "Creates a version selector that runs always."
  (->ver-always))

