; Licensed to the Apache Software Foundation (ASF) under one
; or more contributor license agreements. See the NOTICE file
; distributed with this work for additional information
; regarding copyright ownership. The ASF licenses this file
; to you under the Apache License, Version 2.0 (the
; "License"); you may not use this file except in compliance
; with the License. You may obtain a copy of the License at
;
; http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.

(ns org.domaindrivenarchitecture.pallet.crate.dda-crate-new
  (:require
    [schema.core :as s]  
    [pallet.actions :as actions]
    [pallet.stevedore :as stevedore]
    [pallet.script :as script]
    [pallet.crate :as crate]
    [pallet.api :as api]
    [pallet.node-value :as nv]
    [pallet.core.session :refer [session session! *session*]]
    
    [org.domaindrivenarchitecture.pallet.crate.config :as config]
    [org.domaindrivenarchitecture.pallet.crate.versions :as versions]
    
    [clojure.tools.logging :as logging]
    [clojure.string :refer [join]]
    ))

(defprotocol VersionRange
  "A protocol for resolving upgrade paths of a crate."
  (plan [this remoteversion palletversion args] 
        "Plans actions to upgrade from remoteversion to palletversion. Here remoteversion is a pallet nodevalue."))

(dev UpgradeVector 
     {VersionRange install-fn})

(defrecord VersionedPlan [upgradevec]
  VersionRange
  (plan [this remoteversion palletversion args]
	  (doseq [entry upgradevec]
      (actions/plan-when 
        ; use first entry to decide if actions need to be planed
        (versions/compare-versions (first entry) (nv/node-value remoteversion (session)) palletversion)
        ; run functions of second entry
        ; if second entry is a collection run each function
        (doseq [f (if (coll? (second entry)) (second entry) [(second entry)])] 
          (if (fn? f) (apply f args)))
        ; run for each instance
        (doseq [[instance-key instance-config] (:instances (second args))]
          (doseq [f (if (coll? (nth entry 2 nil)) (nth entry 2 nil) [(nth entry 2 nil)])] 
           (if (fn? f) (apply f (flatten [args instance-key instance-config])))))
  ))))


(defmacro defversionedplan [symb & args]
  "Syntax sugar to create a upgradevector record.
   Usage:

   (defversionpath mypath
      selector-fn plan-fn [optional: instance-plan-fn]
      ...)"
 
  ;`(do (def ~symb (~->VersionedPlan (partition 3 ))))
  `(do (def ~symb (~->VersionedPlan (map flatten (partition 2 (partition-by (fn [_#] (satisfies? versions/VersionSelector _#)) [~@args]))))))
)

(defn- install-marker-path [app]
  "Gets the path of state marker."
  (str "/home/pallet/state/" (name (:facility app))))

(defn- node-write-state [app]
  "Creates an actions that writes a state file to the node."
   (actions/remote-file
      (install-marker-path app) 
      :overwrite-changes true
      :literal true
      :content (versions/ver_str (:version app))))

(defn- node-read-state [app]
  "Read the remote statefile of an app and returns the content as a nodevalue."
  (let [statefile (install-marker-path app)
      nv_version (nv/make-node-value (nv/node-value-symbol))]
	  ; set version to `nil`, if no state file exists
	  (actions/plan-when-not
	    (stevedore/script (file-exists? ~statefile))
	    (actions/as-action (nv/assign-node-value nv_version nil))
	    )
	  ; set version according to state file, if file exists
	  (actions/plan-when
	    (stevedore/script (file-exists? ~statefile))
	    (let [statefilecontent (actions/remote-file-content statefile)]
	      (actions/as-action (nv/assign-node-value nv_version (versions/ver_fromstr @statefilecontent) ))
	      ))   
	  ; log 
	  (actions/as-action (logging/info
	                       "Nodeversion of" (name (:facility app)) "is" (nv/node-value nv_version (session)) ";" 
	                       "Palletversion is" (:version app)))
	  nv_version))

(defprotocol Installable
  "A protocol for a installable dda app."
  
  (create-install-plan [app] "Installs the app.")
  (create-configure-plan [app] "Configures the app.")
  (create-server-spec [app] "Creates a pallet server-spec for the app."))

(defn- runplan [_plan remoteversion palletversion args]
  "Helper function to run a plan. This will... 
     * return nil if plan is nil.
     * call the plan method if the plan implements VersionPath.
     * call the function if plan is a crate function."
  (cond
    (satisfies? VersionPath _plan) ; version path
      (plan _plan remoteversion palletversion args)
    (fn? _plan) ; simple fn
      (apply _plan args)
    )
  )

(defmulti install (fn [dda-crate dda-context config] (:facility dda-crate)))
(defmulti config (fn [dda-crate config] (:facility dda-crate)))
(defmulti preapre-rollout (fn [dda-crate config] (:facility dda-crate)))

(defmethod dda-crate/config :managed-ide [dda-crate config]
  (...))

(defprotocol PlanableFunction 
  (execute [pallet-context :- PalletContext dda-crate :- DdaCrate]))

(defrecord DdaContext [session configuration]
  (get-installed-versions)
  (get-configuration))

(defrecord DdaCrate [facility 
                     version :- Version 
                     schema 
                     default-config]
  ; A record for a versioned dda crate. This implements the Installable protocol 
  ; and requires the facility, current version and two records that implement the 
  ; VersionPath protocol.
  Installable
  (create-install-plan [app]
    (when-selector is-installed?)
    (runplan installplan (node-read-state app) (:version app) [(name (:facility app))  (config/get-nodespecific-additional-config (:facility app))])
    (node-write-state app))
  (create-versioned-install-plan [app [selector install-fn]]
    ...)
  (create-configure-plan [this]
    (api/plan-fn
      (apply config this (config/get-nodespecific-additional-config (:facility this)))))
  (create-server-spec [app] 
    (api/server-spec
      :phases {
        :configure (create-configure-plan app)
        :install (api/plan-fn (create-install-plan app))}
      ))
)

(def create-versioned-crate ->VersionedCrate)
