; Licensed to the Apache Software Foundation (ASF) under one
; or more contributor license agreements. See the NOTICE file
; distributed with this work for additional information
; regarding copyright ownership. The ASF licenses this file
; to you under the Apache License, Version 2.0 (the
; "License"); you may not use this file except in compliance
; with the License. You may obtain a copy of the License at
;
; http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.

(ns org.domaindrivenarchitecture.pallet.crate.dda-versioncrate
  (:require
      [pallet.actions :as actions]
      [pallet.stevedore :as stevedore]
      [pallet.script :as script]
      [pallet.crate :as crate]
      [pallet.api :as api]
      [pallet.node-value :as nv]
      [pallet.core.session :refer [session session!]]
      
      [clojure.tools.logging :as logging]
      [clojure.string :refer [join]]
      [com.palletops.leaven.protocols :refer [Startable Stoppable]]
      ))

; todo: - version auslesen -> settings phase (?)
;       - was ist session/node-values? -> markdown erstellen
;       - clojure testcases erstellen!

(defn- str2int
  "Converts string to integer, ignores all characters but 0-9. 
   Returns 0 if string is empty."
  [str]
  (if (= 0 (count str))
    0
    (Integer. (re-find  #"\d+" str ))))

(defn- ver_fromstr
  "Converts formated version string (1.2.3.4) to version vector [1 2 3 4]"
  [str]
  (into [] (map str2int (clojure.string/split str #"\.")))) ; clj syntax?

(defn- ver_str [version]
  "Converts verstion vector to point seperated string.
   Example (= (ver_str [2 1 4]) \"2.1.4\") "
  (join "." version))

(defn- ver_fill [ver length]
  "Fills up a version vector with trailing zeros.
   Example (= (verfill [2 1] 4) [2 1 4 4]) "
  (if (> length (count ver))
    (concat ver (repeat (- length (count ver)) 0))
    ver
  ))

(defn- ver_comp [v1 v2]
  "Compares two version vectors and return the difference of the first postion they differ.
   Returns nil if they are the same version."
  (let [len (max (count v1) (count v2))]
   (first (drop-while #(= % 0) (mapv - (ver_fill v1 len) (ver_fill v2 len))))
  ))

(defn- ver_less [v1 v2]
  "Returns v1 < v2"
  (let [comp (ver_comp v1 v2)]
    (if (nil? comp) false (< comp 0))))
(defn- ver_lesseq [v1 v2]
  "Returns v1 <= v2"
  (let [comp (ver_comp v1 v2)]
    (if (nil? comp) true (< comp 0))))
(defn- ver_eq [v1 v2]
  "Returns v1 == v2"
  (let [comp (ver_comp v1 v2)]
    (nil? comp)))
    


(defn selector_from_lesseq [verfrom]
  (fn [vremote vpallet] (and (not (ver_eq vremote vpallet))
                             (not (nil? vremote)) 
                             (ver_lesseq vremote verfrom))))

(defn selector_from_to [verfrom verto]
  (fn [vremote vpallet] (and (ver_eq vremote verfrom) (ver_eq vpallet verto))))

(defn selector_clean_install []
  (fn [vremote vpallet] (nil? vremote)))

(defn selector_always []
  (fn [vremote vpallet] true))


(defn- install-marker-path [app-name & [instance-key]]
  "Gets path of state marker"
  (str "/home/pallet/state/" app-name (if instance-key (str "-" (name instance-key)))))

(defn- node-write-state [app-name version & [instance-key]]
  "Creates an actions that writes a state file to the node."
   (actions/remote-file
      (install-marker-path app-name instance-key) 
      :overwrite-changes true
      :literal true
      :content (ver_str version)))


(crate/defplan ^:dynamic install-with-versions
  [app-name version version-set config]
  (let [statefile (install-marker-path app-name)
        nv_version (nv/make-node-value (nv/node-value-symbol))]
    ; set version to `nil`, if no state file exists
    (actions/plan-when-not
      (stevedore/script (file-exists? ~statefile))
      (actions/as-action (nv/assign-node-value nv_version nil))
      )
    ; set version according to state file, if file exists
    (actions/plan-when
      (stevedore/script (file-exists? ~statefile))
      (let [statefilecontent (actions/remote-file-content statefile)]
        (actions/as-action (nv/assign-node-value nv_version (ver_fromstr @statefilecontent) ))
        ))    
    (actions/as-action (logging/info
                         "Nodeversion of" app-name "is" (nv/node-value nv_version (session)) ";" 
                         "Palletversion is" version))
    ; plan actions for each selector
    (doseq [entry version-set]
      (actions/plan-when 
        (apply (:selector entry) [(nv/node-value nv_version (session)) version])
        (doseq [f (:install-fn entry)] (f app-name version config)
      ))
      (node-write-state app-name version))
    ))