(ns dda.backup.restic.domain
  (:require
   [orchestra.core :refer [defn-spec]]
   [clojure.spec.alpha :as s]
   [dda.backup.core.domain :as cd]))

(s/def ::certificate-file string?)
(s/def ::password-file string?)
(s/def ::restic-repository string?)
(s/def ::backup-path string?)
(s/def ::days-to-keep pos?)
(s/def ::months-to-keep pos?)

(s/def ::restic-config
  (s/keys :req-un [::restic-repository 
                   ::backup-path 
                   ::days-to-keep 
                   ::months-to-keep]
          :opt-un [::certificate-file 
                   ::password-file 
                   ::cd/execution-directory]))

(defn-spec repo-command ::cd/command
  [config ::restic-config
   command ::cd/command]
  (let [{:keys [certificate-file password-file execution-directory 
                restic-repository backup-path]} config]
    (into
     []
     (concat
      (if (some? execution-directory)
        [{:dir execution-directory}]
        [])
      ["restic" "-r" (str restic-repository "/" backup-path) "-v"]
      (cond
        (some? certificate-file)
        ["--cacert" certificate-file]
        (some? password-file)
        ["--password-file" password-file]
        :else
        [])
      command))))

(defn-spec check-repo-command ::cd/commands
  [config ::restic-config]
  [(repo-command config ["check"])])

(defn-spec init-repo-command ::cd/commands
  [config ::restic-config]
  [(repo-command config ["init"])])

(defn-spec unlock-repo-command ::cd/commands
  [config ::restic-config]
  [(repo-command config ["--cleanup-cache" "unlock"])])

(defn-spec list-snapshot-command ::cd/commands
  [config ::restic-config]
  [(repo-command config ["snapshots"])])

(defn-spec forget-command ::cd/commands
  [config ::restic-config]
  (let [{:keys [days-to-keep months-to-keep]} config]
    [(repo-command config ["forget" "--group-by" ""
                           "--keep-last" "1"
                           "--keep-daily" (str days-to-keep)
                           "--keep-monthly" (str months-to-keep) "--prune"])]))