(ns dda.clj-threats.domain.score
  (:require
   [clojure.math :as ma]
   [clojure.spec.alpha :as s]
   [orchestra.core :refer [defn-spec]]
   [dda.clj-threats.framework.sanitize :as san]))

(defn score-value? [v] (and (>= v 0) (<= v 99)))
(s/def ::score-value score-value?)
(s/def ::dread-type #{:Damage :Reproducibility :Exploitability :AffectedUsers :Discoverability})
(s/def ::score (s/map-of ::dread-type ::score-value))
(s/def ::il-type #{:impact :likelihood})
(defn il-value? [v] (and (> v 0.0) (<= v 100.0)))
(s/def ::il-value il-value?)
(s/def ::il-score (s/map-of ::il-type ::il-value))



(def min-score {:Reproducibility 1
                :Exploitability 1
                :AffectedUsers 1
                :Discoverability 1
                :Damage 1})

(def max-score {:Reproducibility 99
                :Exploitability 99
                :AffectedUsers 99
                :Discoverability 99
                :Damage 99})

(defn-spec impact-likelihood-score ::il-score
  [dread-score ::score]
  (let [{:keys [Damage Reproducibility Exploitability AffectedUsers Discoverability]} dread-score]
    {:impact (ma/pow (* Damage AffectedUsers) (/ 1 2))
     :likelihood (ma/pow (* Reproducibility Exploitability Discoverability) (/ 1 3))}))

(defn filter-inactive [active-elems childs]
  (reduce-kv
   (fn [result key elem]
     (if (or (not (contains? elem :active_when))
             (san/seq-intersection-some? active-elems (:active_when elem)))
       (merge result {key elem})
       result))
   {}
   childs))


(defn calculate-and [result key and-child]
  (let [child-score (:score and-child)]
    {:Reproducibility (min (:Reproducibility result) (:Reproducibility child-score))
     :Exploitability (min (:Exploitability result) (:Exploitability child-score))
     :AffectedUsers (min (:AffectedUsers result) (:AffectedUsers child-score))
     :Discoverability (min (:Discoverability result) (:Discoverability child-score))
     :Damage (min (:Damage result) (:Damage child-score))}))

(defn calculate-or [result key and-child]
  (let [child-score (:score and-child)]
    {:Reproducibility (max (:Reproducibility result) (:Reproducibility child-score))
     :Exploitability (max (:Exploitability result) (:Exploitability child-score))
     :AffectedUsers (max (:AffectedUsers result) (:AffectedUsers child-score))
     :Discoverability (max (:Discoverability result) (:Discoverability child-score))
     :Damage (max (:Damage result) (:Damage child-score))}))

(defn score-for-leaf [elem]
  (let [is-map? (map? elem)]
    (cond

      (and is-map? (contains? elem :and))
      (assoc elem :score
             (reduce-kv
              calculate-and
              max-score
              (:and elem)))

      (and is-map? (contains? elem :or))
      (assoc elem :score
             (reduce-kv
              calculate-or
              min-score
              (:or elem)))

      (and is-map? (contains? elem :childs))
      (assoc elem :score
             (merge (get-in elem [:childs :score]) (:score elem)))

      (and is-map? (contains? elem :score))
      (assoc elem :score
             (merge min-score (:score elem)))

      :else elem)))

(defn filter-inactive-childs [active-elems elem]
  (let [is-map? (map? elem)]
    (cond

      (and is-map? (contains? elem :and))
      {:and (filter-inactive active-elems (:and elem))}

      (and is-map? (contains? elem :or))
      {:or (filter-inactive active-elems (:or elem))}

      :else elem)))

