(ns dda.clj-threats.domain.report.threats
  (:require
   [clojure.spec.alpha :as s]
   [clojure.string :as st]
   [orchestra.core :refer [defn-spec]]
   [dda.c4k-common.common :as cm]
   [dda.clj-threats.domain.threagile :as ta]
   [dda.clj-threats.domain.attack-tree :as at]
   [dda.clj-threats.domain.score :as sc]
   [dda.clj-threats.domain.rules :as ru]
   [dda.clj-threats.domain.report.model :as mod]))

(defn-spec threat-score (s/* string?)
  [dread-score ::sc/score]
  (let [{:keys [Damage Reproducibility Exploitability AffectedUsers Discoverability]} dread-score
        il-score (sc/impact-likelihood-score dread-score)
        {:keys [impact likelihood]} il-score]
    ["== Score"
     ""
     "|==="
     "| Impact | Likelihood "
     ""
     (str "| " (int impact))
     (str "| " (int likelihood))
     "|==="
     ""
     "|==="
     "| Damage | AffectedUsers | Reproducibility | Exploitability | Discoverability "
     ""
     (str "| " Damage)
     (str "| " AffectedUsers)
     (str "| " Reproducibility)
     (str "| " Exploitability)
     (str "| " Discoverability)
     "|==="
     ""]))

(defn-spec threat-reference (s/* string?)
  [attack ::ru/attack]
  (let [{:keys [cwe owasp-asvs]} attack]
    (if (or (some? cwe) (some? owasp-asvs))
      (cm/concat-vec
       ["== Reference"
        ""]
       (when (some? cwe)
         [(str "- cwe: https://cwe.mitre.org/data/definitions/" cwe ".html[CWE- " cwe "]")])
       (when (some? owasp-asvs)
         [(str "- owasp: " owasp-asvs)])
       [""])
      [])))

(defn-spec threat ::mod/document
  [threagile ::ta/threagile
   attacks ::at/attacks
   threat-entry ::ru/threat]
  (let [id (get-in threat-entry [:attack :id])
        {:keys [data-asset attack link]} threat-entry
        {:keys [score description]} attack
        childs []]

    {:path (str "threat-" (hash id) ".adoc")
     :dir ""
     :lines
     (cm/concat-vec
      [(str "= " id)
       ""]
      (when (not (empty? description))
        [(:description attack)
         ""])
      (threat-score score)
      ["== Example Data Asset affected"
       ""
       (str "Data asset at risk: " (:id data-asset))
       ""
       (str "Communication link: " (:id link))
       ""]
      (threat-reference attack)
      (map mod/include-child childs))
     :childs childs}))

(defn-spec threats ::mod/document
  [threagile ::ta/threagile
   attacks ::at/attacks
   most-important-threats ::ru/threats]
  (let [childs (map #(threat threagile attacks %)  most-important-threats)]
    {:path "threats.adoc"
     :dir ""
     :lines
     (cm/concat-vec
      ["= Threats"
       ""]
      (map mod/include-child childs))
     :childs childs}))