(ns dda.clj-threats.domain.threagile
  (:require
   [clojure.spec.alpha :as s]
   [clojure.walk :as w]
   [orchestra.core :refer [defn-spec]]
   [dda.clj-threats.framework.sanitize :as san]
   [dda.clj-threats.infra.threagile-repo :as repo]
   [dda.clj-threats.domain.threagile.data-asset :as da]
   [dda.clj-threats.domain.threagile.technical-asset :as ta]
   [dda.clj-threats.domain.threagile.boundary :as by]
   [dda.clj-threats.domain.threagile.communication-link :as cl]))

(s/def ::title string?)

(s/def ::threagile
  (s/keys :req-un [::title ::da/data_assets ::ta/technical_assets ::trust_boundaries]
          :opt-un []))
(s/def ::data-asset ::da/data-asset)
(s/def ::technical-asset ::ta/technical-asset)
(s/def ::link ::cl/communication-link)
(s/def ::trust_boundaries ::by/trust_boundaries)
(s/def ::communication (s/keys :req-un [::source ::link ::target]))
(s/def ::communications (s/* ::communication))

(defn-spec get-threagile ::threagile
  [path string?]
  (->>
   (repo/get-threagile path)
   (w/prewalk (fn [x] (san/replace-nil-by-empty-map x :communication_links)))
   (w/prewalk (fn [x] (san/add-id-values x :communication_links)))
   (w/prewalk (fn [x] (san/add-id-values x :data_assets)))
   (w/prewalk (fn [x] (san/add-id-values x :technical_assets)))
   (w/prewalk (fn [x] (san/add-id-values x :trust_boundaries)))))

(defn- calculate-asset-links
  [current-asset all-assets]
  (map
   (fn [link]
     (let [key (first link)
           value (second link)]
       {:source current-asset
        :link value
        :target (get all-assets (keyword (:target value)))}))
   (:communication_links current-asset)))

(defn-spec data-asset-filtered (s/* ::data-asset)
  [names (s/* string?)
   filter-pred fn?
   threagile ::threagile]
  (filter
   filter-pred
   (map
    (fn [x]
      (get-in threagile [:data_assets (keyword x)]))
    names)))

(defn-spec communication-links ::communications
  [threagile ::threagile]
  (let [{:keys [technical_assets]} threagile]
    (->> (map
          #(calculate-asset-links % technical_assets)
          (vals technical_assets))
         (filter not-empty)
         (flatten))))

(defn-spec get-boundary ::by/trust_boundaries
  [ta ::ta/technical-asset
   threagile ::threagile]
  (->> (vals (:trust_boundaries threagile))
       (filter (fn [x] (san/seq-contains? (:technical_assets_inside x) (:id ta))))))

(defn-spec get-technical-asset ::ta/technical-asset
  [asset-id ::ta/id
   threagile ::threagile]
  (->> (vals (:technical_assets threagile))
       (filter (fn [x] (= asset-id (:id x))))))

(defn-spec cross-boundary? boolean?
  [source ::ta/technical-asset
   target ::ta/technical-asset
   threagile ::threagile]
  (not (= (get-boundary source threagile)
          (get-boundary target threagile))))