(ns dda.clj-threats.domain.rules
  (:require
   [clojure.spec.alpha :as s]
   [orchestra.core :refer [defn-spec]]
   [dda.c4k-common.common :as cm]
   [dda.clj-threats.domain.scored-attack-tree :as sat]
   [dda.clj-threats.domain.threagile :as ta]
   [dda.clj-threats.domain.rule.model :as rm]
   [dda.clj-threats.domain.rule.persistent :as rp]
   [dda.clj-threats.domain.rule.communication :as rc]
   [dda.clj-threats.domain.rule.authentication :as ra]
   ))

(s/def ::data-asset-ids ::rm/data-asset-ids)
(s/def ::attack ::sat/node)
(s/def ::threat (s/keys :req-un [::ta/data-asset ::ta/link ::attack]))
(s/def ::threats (s/* ::threat))

(defn-spec crossborder-normalized-communication ::threats
  [source ::ta/technical-asset
   target ::ta/technical-asset
   link ::ta/link
   assets-affected ::data-asset-ids
   threagile ::ta/threagile
   attacks ::sat/attacks]
  [(rc/order-related-assets source target link assets-affected threagile attacks)
   (rc/reference-containing-assets source target link assets-affected threagile attacks)
   (rp/incoming-data-assets source target link assets-affected threagile attacks)
   (ra/verify-incoming-data-assets source target link assets-affected threagile attacks)
   ])

(defn-spec communication-related-threats ::threats
  [comms ::ta/communications
   threagile ::ta/threagile
   attacks ::sat/attacks]
  (cm/concat-vec
   (flatten
    (for [comm comms
          :let [{:keys [source link target]} comm
                assets-received (get-in link [:data_assets_received])
                assets-sent (get-in link [:data_assets_sent])
                is-cross-boundary? (ta/cross-boundary?
                                    source target threagile)]
          :when is-cross-boundary?]
      (cond (and (pos? (count assets-received))
                 (not (get-in source [:out_of_scope])))
            (crossborder-normalized-communication 
             target source link assets-received threagile attacks)
            
            (and (pos? (count assets-sent))
                 (not (get-in target [:out_of_scope])))
            (crossborder-normalized-communication
             source target link assets-sent threagile attacks))))))

(defn-spec score double?
  [threat ::threat] 
  (let [{:keys [Damage Reproducibility Exploitability AffectedUsers Discoverability]} 
        (get-in threat [:attack :score])]
    (* 0.2 (+ Damage Reproducibility Exploitability AffectedUsers Discoverability))))

(defn-spec sort-threats ::threats
  [threats ::threats]
  (sort-by score > threats))

(defn-spec calculate-threats ::threats
  [threagile ::ta/threagile
   attacks ::sat/attacks]
  (sort-threats
   (communication-related-threats
    (ta/communication-links threagile)
    threagile attacks)))