(ns dda.clj-threats.domain.rules
  (:require
   [clojure.spec.alpha :as s]
   [orchestra.core :refer [defn-spec]]
   [dda.c4k-common.common :as cm]
   [dda.clj-threats.domain.threagile.data-asset :as da]
   [dda.clj-threats.domain.scored-attack-tree :as sat]
   [dda.clj-threats.domain.threagile :as ta]
   ))

(s/def ::data-asset-ids (s/* string?))
(s/def ::attack ::sat/node)
(s/def ::threat (s/keys :req-un [::da/data_asset ::ta/link ::attack]))
(s/def ::threats (s/* ::threat))

(defn-spec reference-containing-communication ::threat
  [source ::ta/source
   target ::ta/target
   link ::ta/link
   assets-affected ::data-asset-ids
   threagile ::ta/threagile
   attacks ::sat/attacks]
  (let [activity-pub-assets (ta/data-asset-filtered
                             assets-affected
                             (fn [x] (some #(= "activity-pub" %) (:tags x)))
                             threagile)
        first-asset (first activity-pub-assets)]
    (when (pos? (count activity-pub-assets))
          {:id (str "dos-unintended-proxy-threat: " (:id source) " -> " (:id target))
           :data_asset first-asset
           :link link
           :attack (get-in attacks [:attacks_used :dos-unintended-proxy-or-intermediary])})))

(defn-spec order-related-communication ::threat
  [source ::ta/source
   target ::ta/target
   link ::ta/link
   assets-affected ::data-asset-ids
   threagile ::ta/threagile
   attacks ::sat/attacks]
  (let [in-order-assets (ta/data-asset-filtered
                         assets-affected
                         (fn [x] (some #(= "in-order-activities" %) (:tags x)))
                         threagile)
        first-asset (first in-order-assets)]
    (when (pos? (count in-order-assets))
      {:id (str "out-of-order-threat: " (:id source) " -> " (:id target))
       :data_asset first-asset
       :link link
       :attack (get-in attacks [:attacks_used :activity-out-of-order-attack])})))

(defn-spec crossborder-normalized-communication ::threats
  [source ::ta/source
   target ::ta/target
   link ::ta/link
   assets-affected ::data-asset-ids
   threagile ::ta/threagile
   attacks ::sat/attacks]
  [(order-related-communication source target link assets-affected threagile attacks)
   (reference-containing-communication source target link assets-affected threagile attacks)])

(defn-spec communication-related-threats ::threats
  [comms ::ta/communications
   threagile ::ta/threagile
   attacks ::sat/attacks]
  (cm/concat-vec
   (flatten
    (for [comm comms
          :let [{:keys [source link target]} comm
                assets-received (get-in link [:data_assets_received])
                assets-sent (get-in link [:data_assets_sent])
                is-cross-boundary? (ta/cross-boundary?
                                    source target threagile)]
          :when is-cross-boundary?]
      (cond (and (pos? (count assets-received))
                 (not (get-in source [:out_of_scope])))
            (crossborder-normalized-communication 
             target source link assets-received threagile attacks)
            
            (and (pos? (count assets-sent))
                 (not (get-in target [:out_of_scope])))
            (crossborder-normalized-communication
             source target link assets-sent threagile attacks))))))

(defn-spec score double?
  [threat ::threat] 
  (let [{:keys [Damage Reproducibility Exploitability AffectedUsers Discoverability]} 
        (get-in threat [:attack :score])]
    (* 0.2 (+ Damage Reproducibility Exploitability AffectedUsers Discoverability))))

(defn-spec sort-threats ::threats
  [threats ::threats]
  (sort-by score > threats))

(defn-spec calculate-threats ::threats
  [threagile ::ta/threagile
   attacks ::sat/attacks]
  (sort-threats
   (communication-related-threats
    (ta/communication-links threagile)
    threagile attacks)))