(ns dda.clj-threats.application
  (:gen-class)
  (:require
   [clojure.spec.alpha :as s]
   [clojure.string :as cs]
   [clojure.java.io :as io]
   [clojure.tools.reader.edn :as edn]
   [orchestra.core :refer [defn-spec]]
   [dda.clj-threats.domain.scored-attack-tree :as sa]
   [dda.clj-threats.domain.attack-tree :as ak]
   [dda.clj-threats.domain.threagile :as ta]
   [dda.clj-threats.domain.rules :as ru]
   [dda.clj-threats.domain.report :as re]
   [dda.clj-threats.infra.report-repo :as rr]))

(s/def ::most-important-threats (s/map-of keyword? ::ru/threat))

(defn save-document [document]
  (doseq [x (:childs document)]
    (save-document x))
  (rr/save document))

(defn usage [name]
  (str
   "usage:
  
  options
  -v | --version : Shows project version
  -h | --help    : Shows help
    
  parameters

   " name " [options] [parameters] {your threagile definition file} {your attack-tree file}"))
(s/def ::opts-and-params
  (s/alt
   :help (s/? #{"-h" "--help"})
   :version (s/? #{"-v" "--version"})))
(s/def ::filename (s/and string?
                         #(not (cs/starts-with? % "-"))))
(s/def ::cmd-args (s/cat :opts-and-params ::opts-and-params
                         :args (s/?
                                (s/cat :threagile-file ::filename
                                       :attack-tree-file ::filename))))

(defn invalid-args-msg
  [name spec args]
  (s/explain spec args)
  (println (str "Bad commandline arguments\n" (usage name))))

(defn parse-args
  [args]
  (let [parsed-args (s/conform ::cmd-args args)
        {:keys [opts-and-params args]} parsed-args
        opts-part (cond
                    (= opts-and-params [:help :clojure.spec.alpha/nil])
                    {}
                    (or (= opts-and-params [:help "-h"])
                        (= opts-and-params [:help "--help"]))
                    {:help true}
                    (some #(= :version %) opts-and-params)
                    {:version true}
                    :else
                    {:invalid true})]
    (println parsed-args)
    (merge opts-part args)))

(defn-spec most-important-threats ::most-important-threats
  [threats ::ru/threats]
  (apply merge
         (map
          (fn [x]
            {(keyword (get-in x [:attack :id])) x})
          threats)))

(defn application 
  [threagile-file attack-tree-file]
  (let [threagile (ta/get-threagile threagile-file)
        attacks (ak/get-attacks attack-tree-file)
        mitigations_active (:mitigations_active attacks)
        attacks-with-mitigation (sa/calc-scores mitigations_active attacks)
        attacks-without-mitigation (sa/calc-scores [] attacks)
        threats-with-mitigation (ru/calculate-threats threagile attacks-with-mitigation)
        threats-without-mitigation (ru/calculate-threats threagile attacks-without-mitigation)
        most-important-threats-with-mitigation (vals (most-important-threats threats-with-mitigation))
        most-important-threats-without-mitigation (vals (most-important-threats threats-without-mitigation))
        report (re/report threagile attacks 
                          most-important-threats-with-mitigation 
                          most-important-threats-without-mitigation)]
    (save-document report)))


(defn -main [& cmd-args]
  (let [name "clj-threats"
        parsed-args (parse-args cmd-args)
        {:keys [threagile-file attack-tree-file]} parsed-args]
    (cond
      (contains? parsed-args :invalid)
      (invalid-args-msg name ::cmd-args cmd-args)
      (contains? parsed-args :help)
      (println (usage name))
      (contains? parsed-args :version)
      (println (some-> (io/resource "project.clj") slurp edn/read-string (nth 2)))
      :else
      (application threagile-file attack-tree-file))
  ))