(ns dda.c4k-keycloak.core
  (:require
   [clojure.spec.alpha :as s]
   [orchestra.core :refer [defn-spec]]
   [dda.c4k-common.common :as cm]
   [dda.c4k-common.monitoring :as mon]
   [dda.c4k-common.yaml :as yaml]
   [dda.c4k-keycloak.keycloak :as kc]
   [dda.c4k-keycloak.backup :as backup]
   [dda.c4k-common.ingress :as ing]
   [dda.c4k-common.postgres :as postgres]
   [dda.c4k-common.namespace :as ns]))

(def config-defaults {:namespace "keycloak",
                      :issuer "staging"
                      :db-name "keycloak"
                      :pv-storage-size-gb 30
                      :pvc-storage-class-name :local-path
                      :postgres-image "postgres:14"
                      :postgres-size :2gb
                      :average-rate 100
                      :burst-rate 200})

(s/def ::config (s/keys :req-un [::kc/fqdn]
                        :opt-un [::kc/issuer
                                 ::ns/namespace
                                 ::backup/restic-repository
                                 ::mon/mon-cfg]))

(s/def ::auth (s/keys :req-un [::kc/keycloak-admin-user
                               ::kc/keycloak-admin-password
                               ::postgres/postgres-db-user
                               ::postgres/postgres-db-password]
                      :opt-un [::backup/restic-password
                               ::backup/restic-new-password
                               ::backup/aws-access-key-id
                               ::backup/aws-secret-access-key
                               ::mon/mon-auth]))

(s/def ::config-select (s/* #{"auth" "deployment"}))

(defn-spec config-objects seq?
  [config-select ::config-select
   config ::config]
  (let [resolved-config (merge config-defaults config)
        {:keys [fqdn]} resolved-config
        config-parts (if (empty? config-select)
                       ["auth" "deployment"]
                       config-select)]
    (map yaml/to-string
         (if (some #(= "deployment" %) config-parts)
           (cm/concat-vec
            (ns/generate resolved-config)
            (postgres/config-objects resolved-config)
            (kc/config resolved-config)
            (ing/config-objects (merge
                                 {:service-name "keycloak"
                                  :service-port 80
                                  :fqdns [fqdn]}
                                 resolved-config))
            (when (contains? resolved-config :restic-repository)
              [(backup/generate-config resolved-config)
               (backup/generate-cron)
               (backup/generate-backup-restore-deployment resolved-config)])
            (when (contains? resolved-config :mon-cfg)
              (mon/config-objects resolved-config)))
           []))))

(defn-spec auth-objects seq?
  [config-select ::config-select
   config ::config
   auth ::auth]
  (let [resolved-config (merge config-defaults config)
        config-parts (if (empty? config-select)
                       ["auth" "deployment"]
                       config-select)]
    (map yaml/to-string
         (if (some #(= "auth" %) config-parts)
           (cm/concat-vec
            (postgres/auth-objects resolved-config auth)
            (kc/auth resolved-config auth)
            (when (contains? resolved-config :restic-repository)
              [(backup/generate-secret auth)])
            (when (and (contains? auth :mon-auth) (contains? resolved-config :mon-cfg))
              (mon/auth-objects (:mon-cfg resolved-config) (:mon-auth auth))))
           []))))
