(ns winkel.api
  (:require [ring.util.response :refer [redirect response status]]
            [clojure.tools.logging :as log]
            [durable-queue :as q]
            [clojure.walk :refer [keywordize-keys]]
            [winkel.gateway.paypal :as paypal]
            [winkel.gateway.stripe :as stripe]
            [cheshire.core :as json])
  (:import [java.time ZonedDateTime]
           [java.time.format DateTimeFormatter]
           [java.time.temporal ChronoUnit]))

(defn return-error [resp]
  (let [error (json/parse-string (:body resp) true)]
    (log/fatal error)
    error))

(defn do-request [f xs]
  (let [resp (f xs)
        status (:status resp)]
    (if (some #{status} [200 201])
      (:body resp)
      (return-error resp))))

(defn dispatch-job [queue q xs]
  (try
    (q/put! queue q xs)
    (catch Exception e
      (log/error (.getMessage e)))))

(defmulti session :gateway)
(defmethod session :paypal [plan]
  (do-request paypal/create plan))
(defmethod session :stripe [plan]
  (do-request stripe/session plan))
(defmethod session :wechat [plan]
  (do-request stripe/source plan))
(defmethod session :alipay [plan]
  (do-request stripe/source plan))
(defmethod session :default [plan]
  (session (assoc plan :gateway :stripe)))

(defmulti payment (fn [req _] (keyword (nth (clojure.string/split (:uri req) #"/") 2))))
(defmethod payment :paypal [{params :params :as req} queue]
  (let [resp (paypal/execute (:paymentId params) (:PayerID params))]
    (log/debug "Payment executed with status" (:status resp))
    (if (= (:status resp) 200)
      (let [payment (:body resp)
            payload (case (:state payment)
                      "created" {:message "The transaction was successfully created." :level :success}
                      "approved" (do
                                   (dispatch-job queue :paypal {:event {:event_type "WINKEL.PAYPAL.PAYMENT.APPROVED" :resource payment}})
                                   {:message "The transaction was approved." :level :success})
                      "failed" {:message (:failure_reason payment) :level :danger}
                      "error" {:message (str "We've encountered an error of type " (:message payment) ".") :level :danger})]
        (-> (redirect "/")
            (assoc :flash payload)))
      (return-error resp))))

(defmethod payment :stripe [req queue]
  (let [payload {:message "The transaction was approved." :level :success}]
    (log/info "A payment via Stripe happened.")
    (-> (redirect "/")
        (assoc :flash payload))))

(defmethod payment :alipay [req queue]
  (let [payload {:message "Transaction in flight." :level :success}]
    (log/info "An Alipay payment happened.")
    (-> (redirect "/")
        (assoc :flash payload))))

(defn payment-cancel [_]
  (log/info "Payment was cancelled")
  (-> (redirect "/")
      (assoc :flash {:message "You've cancelled the payment." :level :warning})))

(defn agreement
  "We have to capture the uid here because Paypal does not provide custom field for billing agreements
  https://github.com/paypal/PayPal-REST-API-issues/issues/16"
  [{session :session params :params headers :headers :as req} queue]
  (let [resp (paypal/execute-agreement (:token params))
        agreement (:body resp)
        payload (case (:state agreement)
                  "Active" (do
                             (dispatch-job queue :paypal {:event {:event_type "WINKEL.PAYPAL.AGREEMENT.ACTIVE"
                                                                  :resource (assoc agreement :uid (:uid session))}})
                             {:message "The agreement is active." :level :success})
                  "Cancelled" {:message "The agreement was cancelled." :level :success}
                  "Completed" {:message "The agreement was completed." :level :success}
                  "Created" {:message "The agreement was created." :level :success}
                  "Pending" {:message "The agreement is pending." :level :success}
                  "Reactivated" {:message "The agreement was reactivated." :level :success}
                  "Suspended" {:message "The agreement was suspended." :level :success}
                  (do (log/fatal "No match\n" agreement)
                      {:message "Sorry, there was a problem." :level :danger}))]
    (-> (redirect "/")
        (assoc :flash payload))))

(defn billing-agreement [plan]
  (let [agreement (paypal/create-agreement-definition plan)]
    (do-request paypal/create-agreement agreement)))

(defn agreement-cancel [_]
  (log/info "Agreement approval was cancelled")
  (-> (redirect "/")
      (assoc :flash {:message "You've cancelled the operation." :level :warning})))

(defmulti process-events (fn [req _] (keyword (nth (clojure.string/split (:uri req) #"/") 3))))
(defmethod process-events :paypal [{params :params headers :headers :as req} queue]
  (let [event (keywordize-keys params)]
    (log/info "Dispatching event" (:id event))
    (dispatch-job queue :paypal {:event event}))
  (response ""))

(defmethod process-events :stripe [{params :params headers :headers :as req} queue]
  (let [event (keywordize-keys params)
        signature (get headers "stripe-signature")
        verify (constantly true)]
    (if (verify signature)
      (do (dispatch-job queue :stripe {:event event})
          (response ""))
      (-> (response "")
          (status 400)))))

