(ns winkel.api
  (:require [ring.util.response :refer [redirect response]]
            [clojure.tools.logging :as log]
            [durable-queue :as q]
            [clojure.walk :refer [keywordize-keys]]
            [winkel.gateway.paypal :as paypal]))

(defn dispatch-job [queue resource session params headers type]
  (try
    (q/put! queue :paypal {:resource resource :uid (:uid session) :params params :headers headers :type type})
    (catch Exception e
      (log/error (.getMessage e)))))


(defn payment [{session :session params :params headers :headers :as req} queue]
  (let [payment (paypal/execute (:paymentId params) (:PayerID params))
        payload (case (:state payment)
                  "created" {:message "The transaction was successfully created." :level :success}
                  "approved" (do
                               (dispatch-job queue payment session params headers :payment)
                               {:message "The transaction was approved." :level :success})
                  "failed" {:message (:failure_reason payment) :level :danger}
                  "error" {:message (str "We've encountered an error of type " (:message payment) ".") :level :danger})]
    (-> (redirect "/")
        (assoc :flash payload))))

(defn payment-cancel [_]
  (log/info "Payment was cancelled")
  (-> (redirect "/")
      (assoc :flash {:message "You've cancelled the payment." :level :warning})))

(defn agreement [{session :session params :params headers :headers :as req} queue]
  (let [agreement (paypal/execute-agreement (:token params))
        payload (case (:state agreement)
                  "Active" (do
                             (dispatch-job queue agreement session params headers :agreement)
                             {:message "The agreement is active." :level :success})
                  "Cancelled" {:message "The agreement was cancelled." :level :success}
                  "Completed" {:message "The agreement was completed." :level :success}
                  "Created" {:message "The agreement was created." :level :success}
                  "Pending" {:message "The agreement is pending." :level :success}
                  "Reactivated" {:message "The agreement was reactivated." :level :success}
                  "Suspended" {:message "The agreement was suspended." :level :success})]
    (-> (redirect "/")
        (assoc :flash payload))))

(defn agreement-cancel [_]
  (log/info "Agreement approval was cancelled")
  (-> (redirect "/")
      (assoc :flash {:message "You've cancelled the operation." :level :warning})))

(defn process-events [{session :session params :params headers :headers :as req} queue]
  (let [event (keywordize-keys params)]
    (log/info "Dispatching event" (:id event))
    (dispatch-job queue event session params headers :webhook-event))
  (response ""))
