(ns winkel.api
  (:require [ring.util.response :refer [redirect response]]
            [clojure.tools.logging :as log]
            [durable-queue :as q]
            [winkel.gateway.paypal :as paypal]))

(defn dispatch-payment-job [queue payment session params headers]
  (try
    (q/put! queue :payment {:payment-id (:id payment) :uid (:uid session) :entity-id (:id params) :headers headers})
    (catch Exception e
      (log/error (.getMessage e)))))

(defn dispatch-agreement-job [queue agreement session params headers]
  (try
    (q/put! queue :agreement {:agreement-id (:id agreement) :uid (:uid session) :entity-id (:id params) :headers headers})
    (catch Exception e
      (log/error (.getMessage e)))))

(defn payment [{session :session params :params headers :headers :as req} queue]
  (let [payment (paypal/execute (:paymentId params) (:PayerID params))
        payload (case (:state payment)
                  "created" {:message "The transaction was successfully created." :level :success}
                  "approved" (do (dispatch-payment-job queue payment session params headers)
                               {:message "The transaction was approved." :level :success})
                  "failed" {:message (:failure_reason payment) :level :danger}
                  "error" {:message (str "We've encountered an error of type " (:message payment) ".") :level :danger})]
    (-> (redirect "/")
        (assoc :flash payload))))

(defn payment-cancel [_]
  (log/info "Payment was cancelled")
  (-> (redirect "/")
      (assoc :flash {:message "You've cancelled the payment." :level :warning})))

(defn agreement [{session :session params :params headers :headers :as req} queue]
  (let [agreement (paypal/execute-agreement (:token params))
        payload (case (:state agreement)
                  "Active" (do (dispatch-agreement-job queue agreement session params headers)
                               {:message "The agreement is active." :level :success})
                  "Cancelled" {:message "The agreement was cancelled." :level :success}
                  "Completed" {:message "The agreement was completed." :level :success}
                  "Created" {:message "The agreement was created." :level :success}
                  "Pending" {:message "The agreement is pending." :level :success}
                  "Reactivated" {:message "The agreement was reactivated." :level :success}
                  "Suspended" {:message "The agreement was suspended." :level :success})]
    (-> (redirect "/")
        (assoc :flash payload))))

(defn agreement-cancel [_]
  (log/info "Agreement approval was cancelled")
  (-> (redirect "/")
      (assoc :flash {:message "You've cancelled the operation." :level :warning})))

(defn process-events [{session :session params :params headers :headers :as req} queue]
  (log/info req)
  (let [event (get params "event_type")
        summary (get params "summary")
        resource (get params "resource")]
    (log/info event summary resource (type resource) (contains? resource "billing_agreement_id"))
    (case event
      "PAYMENT.AUTHORIZATION.CREATED" (do)
      "PAYMENT.AUTHORIZATION.VOIDED" (do)
      "PAYMENT.SALE.COMPLETED" (when (contains? resource "billing_agreement_id") (log/info params))
      "PAYMENT.SALE.DENIED" (do)
      "PAYMENT.SALE.PENDING" (do)
      "PAYMENT.SALE.REFUNDED" (do)
      "PAYMENT.SALE.REVERSED" (do)
      "BILLING.PLAN.CREATED" (do)
      "BILLING.PLAN.UPDATED" (do)
      "BILLING.SUBSCRIPTION.CANCELLED" (do)
      "BILLING.SUBSCRIPTION.CREATED" (do)
      "BILLING.SUBSCRIPTION.RE-ACTIVATED" (do)
      "BILLING.SUBSCRIPTION.SUSPENDED" (do)
      "BILLING.SUBSCRIPTION.UPDATED" (do)
      "CUSTOMER.DISPUTE.CREATED" (do)
      "CUSTOMER.DISPUTE.RESOLVED" (do)
      (do)))
  (response ""))
