(ns facetious.authentication.weebly
  (:require [environ.core :refer [env]]
            [ring.util.codec :as codec]
            [ring.util.response :as util]
            [clj-http.client :as client]
            [facetious.helpers :as helpers]
            [kryptos.core :as crypto]
            [clojure.data.json :as json]
            [clj-jwt.core  :refer :all]
            [clojure.string :as str]
            [clojure.tools.logging :as log]))

(def oauth2-params
  {:authorize-uri "https://www.weebly.com/app-center/oauth/authorize" 
   :access-token-uri "https://www.weebly.com/app-center/oauth/access_token"
   :scopes ["read:store-catalog" "read:user" "read:site" "read:membership"]})

(defn app-store [{headers :headers session :session params :params :as req}]
  (let [verified? (fn [req]
                    (let [data (codec/form-encode (select-keys params [:user_id :timestamp :site_id]))
                          key (:weebly-secret env)
                          hmac (:hmac params)]
                      (crypto/verify-hmac key data hmac)))]
    (if (verified? req)
      (util/redirect (str (:callback_url params) "?" (codec/form-encode {:client_id (:weebly-client env)
                                                                         :user_id (:user_id params)
                                                                         :site_id (:site_id params)
                                                                         :redirect_uri (helpers/derive-callback headers "/weebly/callback")
                                                                         :version (:version params)})))
      (do (log/fatal "Could not verify hmac" params)
          (-> (util/response (str "HMAC didn't verify."))
              (util/content-type "text/plain"))))))

(defn callback [{headers :headers session :session params :params :as req}]
  (let [resp (-> (client/post (:callback_url params) {:form-params {:client_id (:weebly-client env)
                                                                    :client_secret (:weebly-secret env)
                                                                    :authorization_code (:authorization_code params)}})
               :body
               (json/read-str :key-fn keyword))
        session (assoc session :weebly {:access_token (:access_token resp)})]
    (-> (util/redirect (:callback_url resp))
        (assoc :session session))))

(defn register [{headers :headers session :session params :params :as req}]
  (let [payload (str->jwt (:jwt params))]
    (if (verify payload (:weebly-secret env))
      (-> (util/redirect "/")
          (assoc :session (assoc-in session [:weebly] (merge (:claims payload) (:weebly session))))
          (assoc :flash {:message "Successfully authorized Weebly." :level :success})))))
